/*
 * File:   mat4j_engines_MatlabEngine.c
 * Author: Gernot WALZL
 */

#include <jni.h>
#include <malloc.h>
#include "engine.h"
#include "mat4j_engines_MatlabEngine.h"
#include "convert.h"

#define BUFSIZE 1023


JNIEXPORT jlong JNICALL Java_mat4j_engines_MatlabEngine_openNative
    (JNIEnv * env, jobject obj, jstring arguments)
{
    const char * args;
    Engine * ep = NULL;

    if (arguments == NULL)
    {
        ep = engOpen("\0");
    }
    else
    {
        args = (*env)->GetStringUTFChars(env, arguments, NULL);
        ep = engOpen(args);
        (*env)->ReleaseStringUTFChars(env, arguments, args);
    }

    return (long)ep;
}


JNIEXPORT void JNICALL Java_mat4j_engines_MatlabEngine_closeNative
    (JNIEnv * env, jobject obj, jlong enginePtr)
{
    Engine * ep = (Engine *)enginePtr;

    if (ep != NULL)
    {
        engClose(ep);
    }
}


JNIEXPORT jlong JNICALL Java_mat4j_engines_MatlabEngine_createOutputBufferNative
    (JNIEnv * env, jobject obj, jlong enginePtr)
{
    Engine * ep = (Engine *)enginePtr;
    char * buffer = NULL;
    int i = 0;

    if (ep == NULL)
    {
        return 0;
    }

    buffer = (char *)malloc(BUFSIZE+1);
    for (i = 0; i <= BUFSIZE; i++)
    {
        buffer[i] = '\0';
    }
    engOutputBuffer(ep, buffer, BUFSIZE);
    buffer[BUFSIZE] = '\0';

    return (long)buffer;
}


JNIEXPORT void JNICALL Java_mat4j_engines_MatlabEngine_destroyOutputBufferNative
    (JNIEnv * env, jobject obj, jlong bufferPtr)
{
    if (bufferPtr != 0)
    {
        free((void *)bufferPtr);
    }
}


JNIEXPORT jboolean JNICALL Java_mat4j_engines_MatlabEngine_evalNative
    (JNIEnv * env, jobject obj, jlong enginePtr, jstring command)
{
    const char * com;
    Engine * ep = (Engine *)enginePtr;
    int error = 0;

    if (ep == NULL)
    {
        return 0;
    }

    com = (*env)->GetStringUTFChars(env, command, NULL);
    error = engEvalString(ep, com);
    (*env)->ReleaseStringUTFChars(env, command, com);

    return (error == 0);
}


JNIEXPORT jstring JNICALL Java_mat4j_engines_MatlabEngine_getOutputNative
    (JNIEnv * env, jobject obj, jlong bufferPtr)
{
    int i = 0;
    char * buffer = (char *)bufferPtr;
    int offset = 2;

    #ifdef WINDOWS
    offset = 0;
    #endif

    jstring result;

    if (buffer == NULL)
    {
        return NULL;
    }

    result = (*env)->NewStringUTF(env, buffer + offset);
    for (i = 0; i <= BUFSIZE; i++)
    {
        buffer[i] = '\0';
    }

    return result;
}


JNIEXPORT jobject JNICALL Java_mat4j_engines_MatlabEngine_getVariableNative
    (JNIEnv * env, jobject obj, jlong enginePtr, jstring name)
{
    const char * var_name;
    jobject result = NULL;
    Engine * ep = (Engine *)enginePtr;
    mxArray * mxa = NULL;

    if (ep == NULL || name == NULL)
    {
        return NULL;
    }

    var_name = (*env)->GetStringUTFChars(env, name, NULL);
    mxa = engGetVariable(ep, var_name);
    (*env)->ReleaseStringUTFChars(env, name, var_name);
    if (mxa != NULL)
    {
        result = mxArray_to_jobject(env, mxa);
        mxDestroyArray(mxa);
    }

    return result;
}


JNIEXPORT jboolean JNICALL Java_mat4j_engines_MatlabEngine_putVariableNative
    (JNIEnv * env, jobject obj, jlong enginePtr, jstring name, jobject value)
{
    const char * var_name;
    Engine * ep = (Engine *)enginePtr;
    mxArray * mxa = NULL;
    int error = 1;

    if (ep == NULL || name == NULL || value == NULL)
    {
        return JNI_FALSE;
    }

    mxa = jobject_to_mxArray(env, value);
    if (mxa != NULL) {
        var_name = (*env)->GetStringUTFChars(env, name, NULL);
        error = engPutVariable(ep, var_name, mxa);
        (*env)->ReleaseStringUTFChars(env, name, var_name);
        mxDestroyArray(mxa);
    }

    return (error == 0);
}


JNIEXPORT jboolean JNICALL Java_mat4j_engines_MatlabEngine_getVisibleNative
    (JNIEnv * env, jobject obj, jlong enginePtr)
{
    Engine * ep = (Engine *)enginePtr;
    bool vis = false;

    if (ep != NULL)
    {
        engGetVisible(ep, &vis);
    }

    return vis;
}


JNIEXPORT void JNICALL Java_mat4j_engines_MatlabEngine_setVisibleNative
    (JNIEnv * env, jobject obj, jlong enginePtr, jboolean vis)
{
    Engine * ep = (Engine *)enginePtr;

    if (ep != NULL)
    {
        engSetVisible(ep, vis);
    }
}
