/*
 * File:   convert.c
 * Author: Gernot WALZL
 */

#include <malloc.h>
#include <jni.h>
#include "engine.h"
#include "convert.h"

#define STRSIZE 1023


jobject mxLogicalArray_to_jbooleanmatrix(JNIEnv * env, const mxArray * mxa)
{
    jclass booleanClass;
    jclass booleanmatrixClass;
    jmethodID booleanInitID;
    jmethodID booleanmatrixInitID;
    jmethodID booleanmatrixSetterID;
    jobject booleanObj = NULL;
    jobject result = NULL;

    int num_rows = 0;
    int num_columns = 0;
    int r = 0;
    int c = 0;
    unsigned char * pr_values = NULL;
    unsigned char r_val = 0;

    booleanClass = (*env)->FindClass(env, "java/lang/Boolean");
    booleanmatrixClass = (*env)->FindClass(env, "mat4j/types/BooleanMatrix");
    if (booleanClass == NULL || booleanmatrixClass == NULL)
    {
        return NULL;
    }
    booleanInitID = (*env)->GetMethodID(env, booleanClass,
            "<init>", "(Z)V");
    booleanmatrixInitID = (*env)->GetMethodID(env, booleanmatrixClass,
            "<init>", "(II)V");
    booleanmatrixSetterID = (*env)->GetMethodID(env, booleanmatrixClass,
            "setValue", "(IILjava/lang/Object;)Z");
    if (booleanInitID == NULL ||
        booleanmatrixInitID == NULL ||
        booleanmatrixSetterID == NULL)
    {
        return NULL;
    }

    num_rows = mxGetM(mxa);
    num_columns = mxGetN(mxa);
    pr_values = mxGetLogicals(mxa);

    result = (*env)->NewObject(env, booleanmatrixClass, booleanmatrixInitID,
            num_rows, num_columns);

    for (r = 0; r < num_rows; r++)
    {
        for (c = 0; c < num_columns; c++)
        {
            r_val = *(pr_values + (r + (c*num_rows)));
            booleanObj = (*env)->NewObject(env, booleanClass, booleanInitID,
                    r_val);
            (*env)->CallBooleanMethod(env, result, booleanmatrixSetterID,
                    r, c, booleanObj);
        }
    }

    (*env)->DeleteLocalRef(env, booleanClass);
    (*env)->DeleteLocalRef(env, booleanmatrixClass);

    return result;
}


jobject mxIntegerArray_to_jintegermatrix(JNIEnv * env, const mxArray * mxa)
{
    jclass integerClass;
    jclass integermatrixClass;
    jmethodID integerInitID;
    jmethodID integermatrixInitID;
    jmethodID integermatrixSetterID;
    jobject integerObj = NULL;
    jobject result = NULL;

    int num_rows = 0;
    int num_columns = 0;
    int r = 0;
    int c = 0;
    int * pr_values = NULL;
    int r_val = 0;

    integerClass = (*env)->FindClass(env, "java/lang/Integer");
    integermatrixClass = (*env)->FindClass(env, "mat4j/types/IntegerMatrix");
    if (integerClass == NULL || integermatrixClass == NULL)
    {
        return NULL;
    }
    integerInitID = (*env)->GetMethodID(env, integerClass,
            "<init>", "(I)V");
    integermatrixInitID = (*env)->GetMethodID(env, integermatrixClass,
            "<init>", "(II)V");
    integermatrixSetterID = (*env)->GetMethodID(env, integermatrixClass,
            "setValue", "(IILjava/lang/Object;)Z");
    if (integerInitID == NULL ||
        integermatrixInitID == NULL ||
        integermatrixSetterID == NULL)
    {
        return NULL;
    }

    num_rows = mxGetM(mxa);
    num_columns = mxGetN(mxa);
    pr_values = mxGetData(mxa);

    result = (*env)->NewObject(env, integermatrixClass, integermatrixInitID,
            num_rows, num_columns);

    for (r = 0; r < num_rows; r++)
    {
        for (c = 0; c < num_columns; c++)
        {
            r_val = *(pr_values + (r + (c*num_rows)));
            integerObj = (*env)->NewObject(env, integerClass, integerInitID,
                    r_val);
            (*env)->CallBooleanMethod(env, result, integermatrixSetterID,
                    r, c, integerObj);
        }
    }

    (*env)->DeleteLocalRef(env, integerClass);
    (*env)->DeleteLocalRef(env, integermatrixClass);

    return result;
}


jobject mxDoubleArray_to_jdoublematrix(JNIEnv * env, const mxArray * mxa)
{
    jclass doubleClass;
    jclass doublematrixClass;
    jmethodID doubleInitID;
    jmethodID doublematrixInitID;
    jmethodID doublematrixSetterID;
    jobject doubleObj = NULL;
    jobject result = NULL;

    int num_rows = 0;
    int num_columns = 0;
    int r = 0;
    int c = 0;
    double * pr_values = NULL;
    double dr_val = 0.0;

    doubleClass = (*env)->FindClass(env, "java/lang/Double");
    doublematrixClass = (*env)->FindClass(env, "mat4j/types/DoubleMatrix");
    if (doubleClass == NULL || doublematrixClass == NULL)
    {
        return NULL;
    }
    doubleInitID = (*env)->GetMethodID(env, doubleClass,
            "<init>", "(D)V");
    doublematrixInitID = (*env)->GetMethodID(env, doublematrixClass,
            "<init>", "(II)V");
    doublematrixSetterID = (*env)->GetMethodID(env, doublematrixClass,
            "setValue", "(IILjava/lang/Object;)Z");
    if (doubleInitID == NULL ||
        doublematrixInitID == NULL ||
        doublematrixSetterID == NULL)
    {
        return NULL;
    }

    num_rows = mxGetM(mxa);
    num_columns = mxGetN(mxa);
    pr_values = mxGetPr(mxa);

    result = (*env)->NewObject(env, doublematrixClass, doublematrixInitID,
            num_rows, num_columns);

    for (r = 0; r < num_rows; r++)
    {
        for (c = 0; c < num_columns; c++)
        {
            dr_val = *(pr_values + (r + (c*num_rows)));
            doubleObj = (*env)->NewObject(env, doubleClass, doubleInitID,
                    dr_val);
            (*env)->CallBooleanMethod(env, result, doublematrixSetterID,
                    r, c, doubleObj);
        }
    }

    (*env)->DeleteLocalRef(env, doubleClass);
    (*env)->DeleteLocalRef(env, doublematrixClass);

    return result;
}


jobject mxComplexArray_to_jcomplexmatrix(JNIEnv * env, const mxArray * mxa)
{
    jclass complexClass;
    jclass complexmatrixClass;
    jmethodID complexInitID;
    jmethodID complexmatrixInitID;
    jmethodID complexmatrixSetterID;
    jobject complexObj = NULL;
    jobject result = NULL;

    int num_rows = 0;
    int num_columns = 0;
    int r = 0;
    int c = 0;
    double * pr_values = NULL;
    double * pi_values = NULL;
    double dr_val = 0.0;
    double di_val = 0.0;

    complexClass = (*env)->FindClass(env, "mat4j/types/Complex");
    complexmatrixClass = (*env)->FindClass(env,"mat4j/types/ComplexMatrix");
    if (complexClass == NULL || complexmatrixClass == NULL)
    {
        return NULL;
    }
    complexInitID = (*env)->GetMethodID(env, complexClass,
            "<init>", "(DD)V");
    complexmatrixInitID = (*env)->GetMethodID(env, complexmatrixClass,
            "<init>", "(II)V");
    complexmatrixSetterID = (*env)->GetMethodID(env, complexmatrixClass,
            "setValue", "(IILjava/lang/Object;)Z");
    if (complexInitID == NULL ||
        complexmatrixInitID == NULL ||
        complexmatrixSetterID == NULL)
    {
        return NULL;
    }

    num_rows = mxGetM(mxa);
    num_columns = mxGetN(mxa);
    pr_values = mxGetPr(mxa);
    pi_values = mxGetPi(mxa);

    result = (*env)->NewObject(env, complexmatrixClass, complexmatrixInitID,
            num_rows, num_columns);

    for (r = 0; r < num_rows; r++)
    {
        for (c = 0; c < num_columns; c++)
        {
            dr_val = *(pr_values + (r + (c*num_rows)));
            di_val = *(pi_values + (r + (c*num_rows)));
            complexObj = (*env)->NewObject(env, complexClass, complexInitID,
                    dr_val, di_val);
            (*env)->CallBooleanMethod(env, result, complexmatrixSetterID,
                    r, c, complexObj);
        }
    }

    (*env)->DeleteLocalRef(env, complexClass);
    (*env)->DeleteLocalRef(env, complexmatrixClass);

    return result;
}


jobject mxCharArray_to_jmatstring(JNIEnv * env, const mxArray * mxa)
{
    jclass matstringClass;
    jmethodID matstringInitID;
    jobject stringObj = NULL;
    jobject result = NULL;

    int i = 0;
    char str[STRSIZE+1];

    matstringClass = (*env)->FindClass(env, "mat4j/types/MatString");
    if (matstringClass == NULL)
    {
        return NULL;
    }
    matstringInitID = (*env)->GetMethodID(env, matstringClass,
            "<init>", "(Ljava/lang/String;)V");
    if (matstringInitID == NULL)
    {
        return NULL;
    }

    for (i = 0; i <= STRSIZE; i++)
    {
        str[i] = '\0';
    }
    mxGetString(mxa, str, STRSIZE);
    stringObj = (*env)->NewStringUTF(env, str);

    result = (*env)->NewObject(env, matstringClass, matstringInitID,
            stringObj);

    (*env)->DeleteLocalRef(env, matstringClass);

    return result;
}


jobject mxCellArray_to_jmatcell(JNIEnv * env, const mxArray * mxa)
{
    jclass matcellClass;
    jmethodID matcellInitID;
    jmethodID matcellSetterID;
    jobject valueObj = NULL;
    jobject result = NULL;

    mxArray * value = NULL;

    int num_elements = 0;
    int i = 0;

    matcellClass = (*env)->FindClass(env, "mat4j/types/MatCell");
    if (matcellClass == NULL)
    {
        return NULL;
    }
    matcellInitID = (*env)->GetMethodID(env, matcellClass,
            "<init>", "(I)V");
    matcellSetterID = (*env)->GetMethodID(env, matcellClass,
            "setValue", "(ILjava/lang/Object;)Z");
    if (matcellInitID == NULL || matcellSetterID == NULL)
    {
        return NULL;
    }

    num_elements = mxGetNumberOfElements(mxa);

    result = (*env)->NewObject(env, matcellClass, matcellInitID, num_elements);

    for (i = 0; i < num_elements; i++)
    {
        value = mxGetCell(mxa, i);
        valueObj = mxArray_to_jobject(env, value);
        (*env)->CallBooleanMethod(env, result, matcellSetterID, i, valueObj);
    }

    (*env)->DeleteLocalRef(env, matcellClass);

    return result;
}


jobject mxStructArray_to_jmatstruct(JNIEnv * env, const mxArray * mxa)
{
    jclass stringClass;
    jclass matstructClass;
    jmethodID matstructInitID;
    jmethodID matstructSetterID;
    jobject stringObj = NULL;
    jobject stringArrayObj = NULL;
    jobject valueObj = NULL;
    jobject result = NULL;

    mxArray * value = NULL;

    int num_elements = 0;
    int num_fields = 0;
    int i = 0;
    int j = 0;
    const char *fieldname;

    stringClass = (*env)->FindClass(env, "java/lang/String");
    matstructClass = (*env)->FindClass(env, "mat4j/types/MatStruct");
    if (stringClass == NULL || matstructClass == NULL)
    {
        return NULL;
    }
    matstructInitID = (*env)->GetMethodID(env, matstructClass,
            "<init>", "([Ljava/lang/String;I)V");
    matstructSetterID = (*env)->GetMethodID(env, matstructClass,
            "setValue", "(Ljava/lang/String;ILjava/lang/Object;)Z");
    if (matstructInitID == NULL || matstructSetterID == NULL)
    {
        return NULL;
    }

    num_elements = mxGetNumberOfElements(mxa);
    num_fields = mxGetNumberOfFields(mxa);

    stringArrayObj = (*env)->NewObjectArray(env, num_fields,
            stringClass, NULL);
    for (i = 0; i < num_fields; i++)
    {
        fieldname = mxGetFieldNameByNumber(mxa, i);
        stringObj = (*env)->NewStringUTF(env, fieldname);
        (*env)->SetObjectArrayElement(env, stringArrayObj, i, stringObj);
    }

    result = (*env)->NewObject(env, matstructClass, matstructInitID,
            stringArrayObj, num_elements);

    for (i = 0; i < num_fields; i++)
    {
        stringObj = (*env)->GetObjectArrayElement(env, stringArrayObj, i);
        for (j = 0; j < num_elements; j++)
        {
            value = mxGetFieldByNumber(mxa, j, i);
            valueObj = mxArray_to_jobject(env, value);
            (*env)->CallBooleanMethod(env, result, matstructSetterID,
                    stringObj, j, valueObj);
        }
        (*env)->DeleteLocalRef(env, stringObj);
    }

    (*env)->DeleteLocalRef(env, stringClass);
    (*env)->DeleteLocalRef(env, matstructClass);

    return result;
}


mxArray * jbooleanmatrix_to_mxLogicalArray(JNIEnv * env, const jobject obj)
{
    jclass booleanClass;
    jclass booleanmatrixClass;
    jmethodID booleanvalueID;
    jmethodID booleanmatrixNumrowsID;
    jmethodID booleanmatrixNumcolumnsID;
    jmethodID booleanmatrixGetterID;
    jobject value;

    mxArray * result = NULL;

    int num_rows = 0;
    int num_columns = 0;
    int r = 0;
    int c = 0;
    unsigned char * pr_values = NULL;
    unsigned char r_val = 0;

    booleanClass = (*env)->FindClass(env, "java/lang/Boolean");
    booleanmatrixClass = (*env)->FindClass(env, "mat4j/types/BooleanMatrix");
    if (booleanClass == NULL ||
        booleanmatrixClass == NULL)
    {
        return NULL;
    }

    booleanvalueID = (*env)->GetMethodID(env, booleanClass,
            "booleanValue", "()Z");
    booleanmatrixNumrowsID = (*env)->GetMethodID(env, booleanmatrixClass,
            "getNumRows", "()I");
    booleanmatrixNumcolumnsID = (*env)->GetMethodID(env, booleanmatrixClass,
            "getNumColumns", "()I");
    booleanmatrixGetterID = (*env)->GetMethodID(env, booleanmatrixClass,
            "getValue", "(II)Ljava/lang/Object;");
    if (booleanvalueID == NULL ||
        booleanmatrixNumrowsID == NULL ||
        booleanmatrixNumcolumnsID == NULL ||
        booleanmatrixGetterID == NULL)
    {
        return NULL;
    }

    num_rows = (*env)->CallIntMethod(env, obj,
            booleanmatrixNumrowsID);
    num_columns = (*env)->CallIntMethod(env, obj,
            booleanmatrixNumcolumnsID);

    result = mxCreateLogicalMatrix(num_rows, num_columns);
    pr_values = mxGetLogicals(result);
    for (r = 0; r < num_rows; r++)
    {
        for (c = 0; c < num_columns; c++)
        {
            value = (*env)->CallObjectMethod(env, obj,
                    booleanmatrixGetterID, r, c);
            if (value == NULL)
            {
                r_val = 0;
            }
            else
            {
                r_val = (*env)->CallBooleanMethod(env, value,
                        booleanvalueID);
            }
            *(pr_values + (r + (c*num_rows))) = r_val;
        }
    }

    (*env)->DeleteLocalRef(env, booleanClass);
    (*env)->DeleteLocalRef(env, booleanmatrixClass);

    return result;
}


mxArray * jintegermatrix_to_mxIntegerArray(JNIEnv * env, const jobject obj)
{
    jclass integerClass;
    jclass integermatrixClass;
    jmethodID integervalueID;
    jmethodID integermatrixNumrowsID;
    jmethodID integermatrixNumcolumnsID;
    jmethodID integermatrixGetterID;
    jobject value;

    mxArray * result = NULL;

    int num_rows = 0;
    int num_columns = 0;
    int r = 0;
    int c = 0;
    int * pr_values = NULL;
    int r_val = 0;

    integerClass = (*env)->FindClass(env, "java/lang/Integer");
    integermatrixClass = (*env)->FindClass(env, "mat4j/types/IntegerMatrix");
    if (integerClass == NULL ||
        integermatrixClass == NULL)
    {
        return NULL;
    }

    integervalueID = (*env)->GetMethodID(env, integerClass,
            "intValue", "()I");
    integermatrixNumrowsID = (*env)->GetMethodID(env, integermatrixClass,
            "getNumRows", "()I");
    integermatrixNumcolumnsID = (*env)->GetMethodID(env, integermatrixClass,
            "getNumColumns", "()I");
    integermatrixGetterID = (*env)->GetMethodID(env, integermatrixClass,
            "getValue", "(II)Ljava/lang/Object;");
    if (integervalueID == NULL ||
        integermatrixNumrowsID == NULL ||
        integermatrixNumcolumnsID == NULL ||
        integermatrixGetterID == NULL)
    {
        return NULL;
    }

    num_rows = (*env)->CallIntMethod(env, obj,
            integermatrixNumrowsID);
    num_columns = (*env)->CallIntMethod(env, obj,
            integermatrixNumcolumnsID);

    result = mxCreateNumericMatrix(num_rows, num_columns,
            mxINT32_CLASS, mxREAL);
    pr_values = mxGetData(result);
    for (r = 0; r < num_rows; r++)
    {
        for (c = 0; c < num_columns; c++)
        {
            value = (*env)->CallObjectMethod(env, obj,
                    integermatrixGetterID, r, c);
            if (value == NULL)
            {
                r_val = 0;
            }
            else
            {
                r_val = (*env)->CallIntMethod(env, value,
                        integervalueID);
            }
            *(pr_values + (r + (c*num_rows))) = r_val;
        }
    }

    (*env)->DeleteLocalRef(env, integerClass);
    (*env)->DeleteLocalRef(env, integermatrixClass);

    return result;
}


mxArray * jdoublematrix_to_mxDoubleArray(JNIEnv * env, const jobject obj)
{
    jclass doubleClass;
    jclass doublematrixClass;
    jmethodID doublevalueID;
    jmethodID doublematrixNumrowsID;
    jmethodID doublematrixNumcolumnsID;
    jmethodID doublematrixGetterID;
    jobject value;

    mxArray * result = NULL;

    int num_rows = 0;
    int num_columns = 0;
    int r = 0;
    int c = 0;
    double * pr_values = NULL;
    double dr_val = 0.0;

    doubleClass = (*env)->FindClass(env, "java/lang/Double");
    doublematrixClass = (*env)->FindClass(env, "mat4j/types/DoubleMatrix");
    if (doubleClass == NULL ||
        doublematrixClass == NULL)
    {
        return NULL;
    }

    doublevalueID = (*env)->GetMethodID(env, doubleClass,
            "doubleValue", "()D");
    doublematrixNumrowsID = (*env)->GetMethodID(env, doublematrixClass,
            "getNumRows", "()I");
    doublematrixNumcolumnsID = (*env)->GetMethodID(env, doublematrixClass,
            "getNumColumns", "()I");
    doublematrixGetterID = (*env)->GetMethodID(env, doublematrixClass,
            "getValue", "(II)Ljava/lang/Object;");
    if (doublevalueID == NULL ||
        doublematrixNumrowsID == NULL ||
        doublematrixNumcolumnsID == NULL ||
        doublematrixGetterID == NULL)
    {
        return NULL;
    }

    num_rows = (*env)->CallIntMethod(env, obj,
            doublematrixNumrowsID);
    num_columns = (*env)->CallIntMethod(env, obj,
            doublematrixNumcolumnsID);

    result = mxCreateNumericMatrix(num_rows, num_columns,
            mxDOUBLE_CLASS, mxREAL);
    pr_values = mxGetPr(result);
    for (r = 0; r < num_rows; r++)
    {
        for (c = 0; c < num_columns; c++)
        {
            value = (*env)->CallObjectMethod(env, obj,
                    doublematrixGetterID, r, c);
            if (value == NULL)
            {
                dr_val = 0.0;
            }
            else
            {
                dr_val = (*env)->CallDoubleMethod(env, value,
                        doublevalueID);
            }
            *(pr_values + (r + (c*num_rows))) = dr_val;
        }
    }

    (*env)->DeleteLocalRef(env, doubleClass);
    (*env)->DeleteLocalRef(env, doublematrixClass);

    return result;
}


mxArray * jcomplexmatrix_to_mxComplexArray(JNIEnv * env, const jobject obj)
{
    jclass complexClass;
    jclass complexmatrixClass;
    jmethodID complexGetreID;
    jmethodID complexGetimID;
    jmethodID complexmatrixNumrowsID;
    jmethodID complexmatrixNumcolumnsID;
    jmethodID complexmatrixGetterID;
    jobject value;

    mxArray *result = NULL;

    int num_rows = 0;
    int num_columns = 0;
    int r = 0;
    int c = 0;
    double *pr_values = NULL;
    double *pi_values = NULL;
    double dr_val = 0.0;
    double di_val = 0.0;

    complexClass = (*env)->FindClass(env, "mat4j/types/Complex");
    complexmatrixClass = (*env)->FindClass(env, "mat4j/types/ComplexMatrix");
    if (complexClass == NULL ||
        complexmatrixClass == NULL)
    {
        return NULL;
    }

    complexGetreID = (*env)->GetMethodID(env, complexClass,
            "getRe", "()D");
    complexGetimID = (*env)->GetMethodID(env, complexClass,
            "getIm", "()D");
    complexmatrixNumrowsID = (*env)->GetMethodID(env, complexmatrixClass,
            "getNumRows", "()I");
    complexmatrixNumcolumnsID = (*env)->GetMethodID(env, complexmatrixClass,
            "getNumColumns", "()I");
    complexmatrixGetterID = (*env)->GetMethodID(env, complexmatrixClass,
            "getValue", "(II)Ljava/lang/Object;");
    if (complexGetreID == NULL ||
        complexGetimID == NULL ||
        complexmatrixNumrowsID == NULL ||
        complexmatrixNumcolumnsID == NULL ||
        complexmatrixGetterID == NULL)
    {
        return NULL;
    }

    num_rows = (*env)->CallIntMethod(env, obj,
            complexmatrixNumrowsID);
    num_columns = (*env)->CallIntMethod(env, obj,
            complexmatrixNumcolumnsID);

    result = mxCreateNumericMatrix(num_rows, num_columns,
            mxDOUBLE_CLASS, mxCOMPLEX);
    pr_values = mxGetPr(result);
    pi_values = mxGetPi(result);
    for (r = 0; r < num_rows; r++)
    {
        for (c = 0; c < num_columns; c++)
        {
            value = (*env)->CallObjectMethod(env, obj,
                    complexmatrixGetterID, r, c);
            if (value == NULL)
            {
                dr_val = 0.0;
                di_val = 0.0;
            }
            else
            {
                dr_val = (*env)->CallDoubleMethod(env, value,
                        complexGetreID);
                di_val = (*env)->CallDoubleMethod(env, value,
                        complexGetimID);
            }
            *(pr_values + (r + (c*num_rows))) = dr_val;
            *(pi_values + (r + (c*num_rows))) = di_val;
        }
    }

    (*env)->DeleteLocalRef(env, complexClass);
    (*env)->DeleteLocalRef(env, complexmatrixClass);

    return result;
}


mxArray * jmatstring_to_mxCharArray(JNIEnv * env, const jobject obj)
{
    jclass matstringClass;
    jmethodID matstringTostringID;
    jobject value = NULL;
    mxArray * result = NULL;
    const char * str;

    matstringClass = (*env)->FindClass(env, "mat4j/types/MatString");
    if (matstringClass == NULL)
    {
        return NULL;
    }
    matstringTostringID = (*env)->GetMethodID(env, matstringClass,
            "toString", "()Ljava/lang/String;");
    if (matstringTostringID == NULL)
    {
        return NULL;
    }
    value = (*env)->CallObjectMethod(env, obj, matstringTostringID);
    str = (*env)->GetStringUTFChars(env, value, NULL);
    result = mxCreateString(str);
    (*env)->ReleaseStringUTFChars(env, value, str);
    (*env)->DeleteLocalRef(env, matstringClass);

    return result;
}


mxArray * jmatcell_to_mxCellArray(JNIEnv * env, const jobject obj)
{
    jclass matcellClass;
    jmethodID matcellGetsizeID;
    jmethodID matcellGetvalueID;
    jobject value = NULL;

    mxArray * val = NULL;
    mxArray * result = NULL;

    int i = 0;
    int size = 0;
    unsigned int dims[1];

    matcellClass = (*env)->FindClass(env, "mat4j/types/MatCell");
    if (matcellClass == NULL)
    {
        return NULL;
    }

    matcellGetsizeID = (*env)->GetMethodID(env, matcellClass,
            "getSize", "()I");
    matcellGetvalueID = (*env)->GetMethodID(env, matcellClass,
            "getValue", "(I)Ljava/lang/Object;");
    if (matcellGetsizeID == NULL ||
        matcellGetvalueID == NULL)
    {
        return NULL;
    }

    size = (*env)->CallIntMethod(env, obj, matcellGetsizeID);
    dims[0] = (unsigned int)size;

    result = mxCreateCellArray(1, dims);
    for (i = 0; i < size; i++)
    {
        value = (*env)->CallObjectMethod(env, obj, matcellGetvalueID, i);
        val = jobject_to_mxArray(env, value);
        mxSetCell(result, i, val);
    }

    (*env)->DeleteLocalRef(env, matcellClass);

    return result;
}


mxArray * jmatstruct_to_mxStructArray(JNIEnv * env, const jobject obj)
{
    jclass matstructClass;
    jmethodID matstructGetfieldnamesID;
    jmethodID matstructGetsizeID;
    jmethodID matstructGetvalueID;
    jobject fieldnamesArrayObj = NULL;
    jobject * fieldnameObjs = NULL;
    jobject value = NULL;

    mxArray * val = NULL;
    mxArray * result = NULL;

    const char** fieldnames;
    int num_fields = 0;
    int i = 0;
    int j = 0;
    int size = 0;
    unsigned int dims[1];

    matstructClass = (*env)->FindClass(env, "mat4j/types/MatStruct");
    if (matstructClass == NULL)
    {
        return NULL;
    }
    matstructGetfieldnamesID = (*env)->GetMethodID(env, matstructClass,
            "getFieldnames", "()[Ljava/lang/String;");
    matstructGetsizeID = (*env)->GetMethodID(env, matstructClass,
            "getSize", "()I");
    matstructGetvalueID = (*env)->GetMethodID(env, matstructClass,
            "getValue", "(Ljava/lang/String;I)Ljava/lang/Object;");
    if (matstructGetfieldnamesID == NULL ||
        matstructGetsizeID == NULL ||
        matstructGetvalueID == NULL)
    {
        return NULL;
    }

    size = (*env)->CallIntMethod(env, obj, matstructGetsizeID);
    dims[0] = (unsigned int)size;

    fieldnamesArrayObj = (*env)->CallObjectMethod(env, obj,
            matstructGetfieldnamesID);
    num_fields = (*env)->GetArrayLength(env, fieldnamesArrayObj);

    fieldnameObjs = (jobject *)malloc(num_fields * sizeof(jobject));
    fieldnames = (const char **)malloc(num_fields * sizeof(const char *));
    for (i = 0; i < num_fields; i++) {
        fieldnameObjs[i] = (*env)->GetObjectArrayElement(env,
                fieldnamesArrayObj, i);
        fieldnames[i] = (*env)->GetStringUTFChars(env, fieldnameObjs[i], NULL);
    }

    result = mxCreateStructArray(1, dims, num_fields, fieldnames);

    for (i = 0; i < size; i++)
    {
        for (j = 0; j < num_fields; j++)
        {
            value = (*env)->CallObjectMethod(env, obj, matstructGetvalueID,
                    fieldnameObjs[j], i);
            val = jobject_to_mxArray(env, value);
            mxSetField(result, i, fieldnames[j], val);
        }
    }

    for (i = 0; i < num_fields; i++)
    {
        (*env)->ReleaseStringUTFChars(env, fieldnameObjs[i], fieldnames[i]);
        (*env)->DeleteLocalRef(env, fieldnameObjs[i]);
    }
    free((void *)fieldnameObjs);
    (*env)->DeleteLocalRef(env, matstructClass);

    return result;
}


jobject mxArray_to_jobject(JNIEnv * env, const mxArray * mxa)
{
    jobject result = NULL;

    if (env == NULL || mxa == NULL)
    {
        return NULL;
    }

    if (mxIsLogical(mxa))
    {
        result = mxLogicalArray_to_jbooleanmatrix(env, mxa);
    }
    else if (mxIsInt32(mxa))
    {
        result = mxIntegerArray_to_jintegermatrix(env, mxa);
    }
    else if (mxIsComplex(mxa))
    {
        result = mxComplexArray_to_jcomplexmatrix(env, mxa);
    }
    else if (mxIsDouble(mxa))
    {
        result = mxDoubleArray_to_jdoublematrix(env, mxa);
    }
    else if (mxIsChar(mxa))
    {
        result = mxCharArray_to_jmatstring(env, mxa);
    }
    else if (mxIsCell(mxa))
    {
        result = mxCellArray_to_jmatcell(env, mxa);
    }
    else if (mxIsStruct(mxa))
    {
        result = mxStructArray_to_jmatstruct(env, mxa);
    }
    else
    {
        printf("Matlab to Java: Class '");
        printf(mxGetClassName(mxa));
        printf("' is not supported.\n");
    }

    return result;
}


mxArray * jobject_to_mxArray(JNIEnv * env, const jobject obj)
{
    jclass booleanmatrixClass;
    jclass integermatrixClass;
    jclass doublematrixClass;
    jclass complexmatrixClass;
    jclass matstringClass;
    jclass matcellClass;
    jclass matstructClass;
    mxArray * result = NULL;

    if (env == NULL || obj == NULL)
    {
        return NULL;
    }

    booleanmatrixClass = (*env)->FindClass(env, "mat4j/types/BooleanMatrix");
    integermatrixClass = (*env)->FindClass(env, "mat4j/types/IntegerMatrix");
    doublematrixClass = (*env)->FindClass(env, "mat4j/types/DoubleMatrix");
    complexmatrixClass = (*env)->FindClass(env, "mat4j/types/ComplexMatrix");
    matstringClass = (*env)->FindClass(env, "mat4j/types/MatString");
    matcellClass = (*env)->FindClass(env, "mat4j/types/MatCell");
    matstructClass = (*env)->FindClass(env, "mat4j/types/MatStruct");

    if ((*env)->IsInstanceOf(env, obj, booleanmatrixClass))
    {
        result = jbooleanmatrix_to_mxLogicalArray(env, obj);
    }
    else if ((*env)->IsInstanceOf(env, obj, integermatrixClass))
    {
        result = jintegermatrix_to_mxIntegerArray(env, obj);
    }
    else if ((*env)->IsInstanceOf(env, obj, doublematrixClass))
    {
        result = jdoublematrix_to_mxDoubleArray(env, obj);
    }
    else if ((*env)->IsInstanceOf(env, obj, complexmatrixClass))
    {
        result = jcomplexmatrix_to_mxComplexArray(env, obj);
    }
    else if ((*env)->IsInstanceOf(env, obj, matstringClass))
    {
        result = jmatstring_to_mxCharArray(env, obj);
    }
    else if ((*env)->IsInstanceOf(env, obj, matcellClass))
    {
        result = jmatcell_to_mxCellArray(env, obj);
    }
    else if ((*env)->IsInstanceOf(env, obj, matstructClass))
    {
        result = jmatstruct_to_mxStructArray(env, obj);
    }
    else
    {
        printf("Java to Matlab: Class is not supported.\n");
    }

    (*env)->DeleteLocalRef(env, booleanmatrixClass);
    (*env)->DeleteLocalRef(env, integermatrixClass);
    (*env)->DeleteLocalRef(env, doublematrixClass);
    (*env)->DeleteLocalRef(env, complexmatrixClass);
    (*env)->DeleteLocalRef(env, matstringClass);
    (*env)->DeleteLocalRef(env, matcellClass);
    (*env)->DeleteLocalRef(env, matstructClass);

    return result;
}
