package mat4j.engines;

import mat4j.types.BooleanMatrix;
import mat4j.types.Complex;
import mat4j.types.ComplexMatrix;
import mat4j.types.DoubleMatrix;
import mat4j.types.IntegerMatrix;
import mat4j.types.MatCell;
import mat4j.types.MatObj;
import mat4j.types.MatString;
import mat4j.types.MatStruct;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import static org.junit.Assert.*;

/**
 * @author Gernot WALZL
 */
public class MatlabEngineTest {

    private static MatlabEngine matlabEngine;

    @BeforeClass
    public static void setUpClass() throws Exception {
        matlabEngine = new MatlabEngine();
        matlabEngine.open(null);
    }

    @AfterClass
    public static void tearDownClass() throws Exception {
        matlabEngine.close();
    }

    @Test
    public void testEval() {
        System.out.println("eval");
        String command = "Mat4J_x = [1:10];";
        boolean result = matlabEngine.eval(command);
        assertEquals(true, result);
    }

    @Test
    public void testGetOutput() {
        System.out.println("getOutput");
        String command = "Mat4J_x = 1+2*3";
        matlabEngine.eval(command);
        String result = matlabEngine.getOutput();
        assertTrue(result.contains("Mat4J_x ="));
        assertTrue(result.contains("7"));
    }

    @Test
    public void testGetVariable() {
        System.out.println("getVariable");
        MatObj result = null;
        int size = 0;
        Object value = null;

        matlabEngine.eval("Mat4J_x = 0==0;");
        result = matlabEngine.getVariable("Mat4J_x");
        assertTrue(result instanceof BooleanMatrix);
        BooleanMatrix booleanMatrix = (BooleanMatrix)result;
        assertEquals(1, booleanMatrix.getNumRows());
        assertEquals(1, booleanMatrix.getNumColumns());
        assertTrue(booleanMatrix.getValue(0, 0).booleanValue());

        matlabEngine.eval("Mat4J_x = int32([1 2; 3 4; 5 6]);");
        result = matlabEngine.getVariable("Mat4J_x");
        assertTrue(result instanceof IntegerMatrix);
        IntegerMatrix integerMatrix = (IntegerMatrix)result;
        assertEquals(3, integerMatrix.getNumRows());
        assertEquals(2, integerMatrix.getNumColumns());
        assertEquals(new Integer(2), integerMatrix.getValue(0, 1));
        assertEquals(new Integer(3), integerMatrix.getValue(1, 0));
        assertEquals(new Integer(6), integerMatrix.getValue(2, 1));

        matlabEngine.eval("Mat4J_x = ones(2,2);");
        result = matlabEngine.getVariable("Mat4J_x");
        assertTrue(result instanceof DoubleMatrix);
        DoubleMatrix doubleMatrix = (DoubleMatrix)result;
        assertEquals(2, doubleMatrix.getNumRows());
        assertEquals(2, doubleMatrix.getNumColumns());
        assertEquals(new Double(1.0), doubleMatrix.getValue(0, 1));

        matlabEngine.eval("Mat4J_x = 1+2i;");
        result = matlabEngine.getVariable("Mat4J_x");
        assertTrue(result instanceof ComplexMatrix);
        ComplexMatrix complexMatrix = (ComplexMatrix)result;
        assertEquals(1, complexMatrix.getNumRows());
        assertEquals(1, complexMatrix.getNumColumns());
        assertEquals(new Complex(1.0, 2.0), complexMatrix.getValue(0, 0));

        matlabEngine.eval("Mat4J_x = 'just testing';");
        result = matlabEngine.getVariable("Mat4J_x");
        assertTrue(result instanceof MatString);
        MatString matString = (MatString)result;
        assertEquals("just testing", matString.toString());

        matlabEngine.eval("Mat4J_x = {'foo', 'bar', [1,2]};");
        result = matlabEngine.getVariable("Mat4J_x");
        assertTrue(result instanceof MatCell);
        MatCell matCell = (MatCell)result;
        size = matCell.getSize();
        assertEquals(3, size);
        value = matCell.getValue(1);
        assertEquals(new MatString("bar"), value);

        matlabEngine.eval("Mat4J_x = struct('testing', {1, 2});");
        result = matlabEngine.getVariable("Mat4J_x");
        assertTrue(result instanceof MatStruct);
        MatStruct matStruct = (MatStruct)result;
        String[] fieldnames = matStruct.getFieldnames();
        assertEquals(new String[]{"testing"}, fieldnames);
        size = matStruct.getSize();
        assertEquals(2, size);
        value = matStruct.getValue(fieldnames[0], 0);
        assertTrue(value instanceof DoubleMatrix);
        Double val = ((DoubleMatrix)value).getValue(0, 0);
        assertEquals(new Double(1.0), val);

        matlabEngine.eval("clear Mat4J_x;");
    }

    @Test
    public void testPutVariable() {
        System.out.println("putVariable");

        BooleanMatrix booleanMatrix = new BooleanMatrix(2, 2);
        booleanMatrix.setValue(0, 0, new Boolean(true));
        matlabEngine.putVariable("Mat4J_x", booleanMatrix);
        assertEquals(booleanMatrix, matlabEngine.getVariable("Mat4J_x"));

        IntegerMatrix integerMatrix = new IntegerMatrix(2, 2);
        integerMatrix.setValue(0, 0, new Integer(1));
        matlabEngine.putVariable("Mat4J_x", integerMatrix);
        assertEquals(integerMatrix, matlabEngine.getVariable("Mat4J_x"));

        DoubleMatrix doubleMatrix = new DoubleMatrix(2, 2);
        doubleMatrix.setValue(0, 0, new Double(1.0));
        matlabEngine.putVariable("Mat4J_x", doubleMatrix);
        assertEquals(doubleMatrix, matlabEngine.getVariable("Mat4J_x"));

        ComplexMatrix complexMatrix = new ComplexMatrix(2, 2);
        complexMatrix.setValue(0, 0, new Complex(1.0, 1.0));
        matlabEngine.putVariable("Mat4J_x", complexMatrix);
        assertEquals(complexMatrix, matlabEngine.getVariable("Mat4J_x"));

        MatString matString = new MatString("just testing");
        matlabEngine.putVariable("Mat4J_x", matString);
        assertEquals(matString, matlabEngine.getVariable("Mat4J_x"));

        MatCell matCell = new MatCell(2);
        matCell.setValue(0, new MatString("foo"));
        matCell.setValue(1, new MatString("bar"));
        matlabEngine.putVariable("Mat4J_x", matCell);
        assertEquals(matCell, matlabEngine.getVariable("Mat4J_x"));

        MatStruct matStruct = new MatStruct(new String[]{"foo", "bar"}, 2);
        matStruct.setValue("bar", 1, new MatString("just testing"));
        matlabEngine.putVariable("Mat4J_x", matStruct);
        assertEquals(matStruct, matlabEngine.getVariable("Mat4J_x"));

        matlabEngine.eval("clear Mat4J_x;");
    }

}
