package mat4j.engines;

import java.awt.image.BufferedImage;
import mat4j.types.BooleanMatrix;
import mat4j.types.DoubleMatrix;
import mat4j.types.IntegerMatrix;
import mat4j.types.MatObj;

/**
 * @author Gernot WALZL
 */
public class MatlabEngine extends MatEngine {

    static {
        System.loadLibrary("mat4j_matlabengine");
    }

    private static final String MSG_NOT_OPENED =
            "Matlab engine is not opened.";
    private static final String MSG_NO_OUTPUT_BUFFER =
            "Output buffer is not initialized.";

    private long enginePtr = 0;
    private long bufferPtr = 0;

    private native long openNative(String arguments);
    private native void closeNative(long enginePtr);
    private native long createOutputBufferNative(long enginePtr);
    private native void destroyOutputBufferNative(long bufferPtr);
    private native boolean evalNative(long enginePtr, String command);
    private native String getOutputNative(long bufferPtr);
    private native MatObj getVariableNative(long enginePtr, String name);
    private native boolean putVariableNative(long enginePtr, String name,
            MatObj value);
    private native boolean getVisibleNative(long enginePtr);
    private native void setVisibleNative(long enginePtr, boolean vis);

    public MatlabEngine() {
        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                close();
            }
        });
    }

    @Override
    public boolean open(String arguments) {
        if (enginePtr != 0) {
            close();
        }
        enginePtr = openNative(arguments);
        if (enginePtr != 0) {
            bufferPtr = createOutputBufferNative(enginePtr);
        }
        return (enginePtr != 0);
    }

    @Override
    public void close() {
        if (enginePtr != 0) {
            closeNative(enginePtr);
            enginePtr = 0;
        }
        if (bufferPtr != 0) {
            destroyOutputBufferNative(bufferPtr);
            bufferPtr = 0;
        }
    }

    @Override
    public boolean eval(String command) {
        if (enginePtr == 0) {
            throw new NullPointerException(MSG_NOT_OPENED);
        }
        return evalNative(enginePtr, command);
    }

    @Override
    public String getOutput() {
        if (bufferPtr == 0) {
            throw new NullPointerException(MSG_NO_OUTPUT_BUFFER);
        }
        return getOutputNative(bufferPtr);
    }

    @Override
    public MatObj getVariable(String name) {
        if (enginePtr == 0) {
            throw new NullPointerException(MSG_NOT_OPENED);
        }
        return getVariableNative(enginePtr, name);
    }

    @Override
    public boolean putVariable(String name, MatObj value) {
        if (enginePtr == 0) {
            throw new NullPointerException(MSG_NOT_OPENED);
        }
        return putVariableNative(enginePtr, name, value);
    }

    /**
     * MS Windows only
     */
    public boolean getVisible() {
        if (enginePtr == 0) {
            throw new NullPointerException(MSG_NOT_OPENED);
        }
        return getVisibleNative(enginePtr);
    }

    /**
     * MS Windows only
     */
    public void setVisible(boolean vis) {
        if (enginePtr == 0) {
            throw new NullPointerException(MSG_NOT_OPENED);
        }
        setVisibleNative(enginePtr, vis);
    }

    /**
     * @param hFigureVarName   variable name of matlabs figure handle
     */
    public BufferedImage getImage(String hFigureVarName) {
        if (enginePtr == 0) {
            throw new NullPointerException(MSG_NOT_OPENED);
        }
        BufferedImage image = null;
        eval("Mat4J_ishandle = ishandle("+hFigureVarName+");");
        MatObj isHandle = getVariable("Mat4J_ishandle");
        eval("clear Mat4J_ishandle;");
        if (isHandle == null) {
            return null;
        }
        if (((BooleanMatrix)isHandle).getValue(0, 0).booleanValue()) {
            eval("Mat4J_w = hardcopy("+hFigureVarName+", '-dOpenGL', '-r0');");

            eval("Mat4J_w_size = size(Mat4J_w);");
            DoubleMatrix size = (DoubleMatrix)getVariable("Mat4J_w_size");
            eval("clear Mat4J_w_size;");

            int height = size.getValue(0, 0).intValue();
            int width = size.getValue(0, 1).intValue();
            image = new BufferedImage(width, height,
                BufferedImage.TYPE_INT_RGB);

            eval("Mat4J_w_r = int32(Mat4J_w(:,:,1));");
            IntegerMatrix red = (IntegerMatrix)getVariable("Mat4J_w_r");
            eval("clear Mat4J_w_r;");

            eval("Mat4J_w_g = int32(Mat4J_w(:,:,2));");
            IntegerMatrix green = (IntegerMatrix)getVariable("Mat4J_w_g");
            eval("clear Mat4J_w_g;");

            eval("Mat4J_w_b = int32(Mat4J_w(:,:,3));");
            IntegerMatrix blue = (IntegerMatrix)getVariable("Mat4J_w_b");
            eval("clear Mat4J_w_b;");

            eval("clear Mat4J_w;");

            int rgb = 0;
            for (int h = 0; h < height; h++) {
                for (int w = 0; w < width; w++) {
                    rgb = red.getValue(h, w).intValue();
                    rgb <<= 8;
                    rgb = rgb | green.getValue(h, w).intValue();
                    rgb <<= 8;
                    rgb = rgb | blue.getValue(h, w).intValue();
                    image.setRGB(w, h, rgb);
                }
            }
        }
        return image;
    }

}
