/*
 * SelfOrganizingMapTest.java
 * JUnit based test
 *
 * Created on 2007-03-23, 13:03
 */

package iaik.som;

import junit.framework.*;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

/**
 * JUnit based test for SelfOragnizingMap.
 *
 * @author Gernot WALZL
 */
public class SelfOrganizingMapTest extends TestCase {

    public SelfOrganizingMapTest(String testName) {
        super(testName);
    }

    public void testReadCodWriteCod() {
        System.out.println("readCod and writeCod");

        SelfOrganizingMap sMap = new SelfOrganizingMap();
        SelfOrganizingMap sMap2 = new SelfOrganizingMap();
        FileInputStream fis = null;
        FileOutputStream fos = null;
        try {
            fis = new FileInputStream("sMap_junit.cod");
            sMap.readCod(fis);
            fis.close();

            fos = new FileOutputStream("sMap_junit_tested.cod");
            sMap.writeCod(fos);
            fos.close();

            fis = new FileInputStream("sMap_junit_tested.cod");
            sMap2.readCod(fis);
            fis.close();
        }
        catch(IOException ex) {
            ex.printStackTrace();
            fail("Test case failed: IOException");
        }

        assertEquals(sMap.toString(), sMap2.toString());
    }

    public void testSetup() {
        System.out.println("setup");

        double[][] trainingData = {{0.7036, 0.1400},
                {0.4850, 0.5668},
                {0.1146, 0.8230},
                {0.6649, 0.6739},
                {0.3654, 0.9994}};
        SelfOrganizingMap sMap = new SelfOrganizingMap();
        sMap.setup(trainingData);

        assertEquals(6, sMap.getData().height);
        assertEquals(2, sMap.getData().width);
    }

    public void testLininit() {
        System.out.println("lininit");

        double[][] trainingData = {{0.7036, 0.1400},
                {0.4850, 0.5668},
                {0.1146, 0.8230},
                {0.6649, 0.6739},
                {0.3654, 0.9994}};
        SelfOrganizingMap sMap = new SelfOrganizingMap();
        sMap.setup(trainingData);
        sMap.lininit(trainingData);
        double[][] cod = sMap.getData().codebook;
        double[][] expected = {{0.7619, 0.434},
                {0.689, 0.5461},
                {0.616, 0.6581},
                {0.5431, 0.7701},
                {0.4701, 0.8821},
                {0.3972, 0.9942},
                {0.5362, 0.2871},
                {0.4633, 0.3991},
                {0.3903, 0.5111},
                {0.3174, 0.6231},
                {0.2444, 0.7352},
                {0.1715, 0.8472}};

        assertTrue(MyMath.equalMatrix(expected, cod, 4));
    }

    public void testUnitCoords() {
        System.out.println("unitCoords");

        SelfOrganizingMap sMap = new SelfOrganizingMap(
                SelfOrganizingMap.Lattice.RECT,
                2, 4, 5, SelfOrganizingMap.Neighborhood.GAUSSIAN);
        double[][] coords;

        coords = sMap.unitCoords(SelfOrganizingMap.Lattice.RECT);
        double[][] expectedrect = {{0.0, 0.0},
                {0.0, 1.0},
                {0.0, 2.0},
                {0.0, 3.0},
                {1.0, 0.0},
                {1.0, 1.0},
                {1.0, 2.0},
                {1.0, 3.0},
                {2.0, 0.0},
                {2.0, 1.0},
                {2.0, 2.0},
                {2.0, 3.0},
                {3.0, 0.0},
                {3.0, 1.0},
                {3.0, 2.0},
                {3.0, 3.0},
                {4.0, 0.0},
                {4.0, 1.0},
                {4.0, 2.0},
                {4.0, 3.0}};

        assertTrue(MyMath.equalMatrix(expectedrect, coords, 0));

        coords = sMap.unitCoords(SelfOrganizingMap.Lattice.HEXA);
        double[][] expectedhexa = {{0.0, 0.0},
                {0.5, 0.866},
                {0.0, 1.732},
                {0.5, 2.598},
                {1.0, 0.0},
                {1.5, 0.866},
                {1.0, 1.732},
                {1.5, 2.598},
                {2.0, 0.0},
                {2.5, 0.866},
                {2.0, 1.732},
                {2.5, 2.598},
                {3.0, 0.0},
                {3.5, 0.866},
                {3.0, 1.732},
                {3.5, 2.598},
                {4.0, 0.0},
                {4.5, 0.866},
                {4.0, 1.732},
                {4.5, 2.598}};

        assertTrue(MyMath.equalMatrix(expectedhexa, coords, 3));
    }

    public void testUnitDists() {
        System.out.println("unitDists");

        SelfOrganizingMap sMap = new SelfOrganizingMap(
                SelfOrganizingMap.Lattice.RECT,
                2, 3, 4, SelfOrganizingMap.Neighborhood.GAUSSIAN);
        double[][] Ud = sMap.unitDists();
        double[][] expected =
            {{0.0, 1.0, 2.0, 1.0, 1.41, 2.24, 2.0, 2.24, 2.83, 3.0, 3.16, 3.61},
            {1.0, 0.0, 1.0, 1.41, 1.0, 1.41, 2.24, 2.0, 2.24, 3.16, 3.0, 3.16},
            {2.0, 1.0, 0.0, 2.24, 1.41, 1.0, 2.83, 2.24, 2.0, 3.61, 3.16, 3.0},
            {1.0, 1.41, 2.24, 0.0, 1.0, 2.0, 1.0, 1.41, 2.24, 2.0, 2.24, 2.83},
            {1.41, 1.0, 1.41, 1.0, 0.0, 1.0, 1.41, 1.0, 1.41, 2.24, 2.0, 2.24},
            {2.24, 1.41, 1.0, 2.0, 1.0, 0.0, 2.24, 1.41, 1.0, 2.83, 2.24, 2.0},
            {2.0, 2.24, 2.83, 1.0, 1.41, 2.24, 0.0, 1.0, 2.0, 1.0, 1.41, 2.24},
            {2.24, 2.0, 2.24, 1.41, 1.0, 1.41, 1.0, 0.0, 1.0, 1.41, 1.0, 1.41},
            {2.83, 2.24, 2.0, 2.24, 1.41, 1.0, 2.0, 1.0, 0.0, 2.24, 1.41, 1.0},
            {3.0, 3.16, 3.61, 2.0, 2.24, 2.83, 1.0, 1.41, 2.24, 0.0, 1.0, 2.0},
            {3.16, 3.0, 3.16, 2.24, 2.0, 2.24, 1.41, 1.0, 1.41, 1.0, 0.0, 1.0},
            {3.61, 3.16, 3.0, 2.83, 2.24, 2.0, 2.24, 1.41, 1.0, 2.0, 1.0, 0.0}};

        assertTrue(MyMath.equalMatrix(expected, Ud, 2));
    }

    public void testBatchtrain() {
        System.out.println("batchtrain");

        double[][] trainingData = {{0.7036, 0.1400},
                {0.4850, 0.5668},
                {0.1146, 0.8230},
                {0.6649, 0.6739},
                {0.3654, 0.9994}};

        SelfOrganizingMap sMap = new SelfOrganizingMap();
        sMap.setup(trainingData);
        sMap.lininit(trainingData);

        TrainingSetupData sTrain = sMap.generateTrainingSetup(
                trainingData.length, SelfOrganizingMap.Training.DEFAULT,
                Double.NaN);
        sMap.batchtrain(sTrain, trainingData);
        double[][] cod = sMap.getData().codebook;
        double[][] expected = {{0.664, 0.3234},
                {0.6171, 0.4613},
                {0.5981, 0.6034},
                {0.5093, 0.6812},
                {0.3886, 0.855},
                {0.2818, 0.9214},
                {0.6638, 0.2634},
                {0.6042, 0.4045},
                {0.5607, 0.5618},
                {0.4422, 0.6708},
                {0.312, 0.8408},
                {0.2174, 0.8822}};

        assertTrue(MyMath.equalMatrix(expected, cod, 4));
    }

    public void testMake() {
        System.out.println("constructor (som_make)");

        double[][] trainingData = {{0.7036, 0.1400},
                {0.4850, 0.5668},
                {0.1146, 0.8230},
                {0.6649, 0.6739},
                {0.3654, 0.9994}};
        SelfOrganizingMap sMap = new SelfOrganizingMap(trainingData);
        double[][] cod = sMap.getData().codebook;
        double[][] expected = {{0.664, 0.3234},
                {0.6171, 0.4613},
                {0.5981, 0.6034},
                {0.5093, 0.6812},
                {0.3886, 0.855},
                {0.2818, 0.9214},
                {0.6638, 0.2634},
                {0.6042, 0.4045},
                {0.5607, 0.5618},
                {0.4422, 0.6708},
                {0.312, 0.8408},
                {0.2174, 0.8822}};

        assertTrue(MyMath.equalMatrix(expected, cod, 4));
    }

    public void testBmus() {
        System.out.println("bmus");

        double[][] trainingData = {{0.7036, 0.1400},
                {0.4850, 0.5668},
                {0.1146, 0.8230},
                {0.6649, 0.6739},
                {0.3654, 0.9994}};
        SelfOrganizingMap sMap = new SelfOrganizingMap(trainingData);
        int[] bmus = sMap.bmus(trainingData);

        assertEquals(6, bmus[0]);
        assertEquals(5, bmus[4]);
    }

    public void testQerrors() {
        System.out.println("qerrors");

        double[][] trainingData = {{0.7036, 0.1400},
                {0.4850, 0.5668},
                {0.1146, 0.8230},
                {0.6649, 0.6739},
                {0.3654, 0.9994}};
        SelfOrganizingMap sMap = new SelfOrganizingMap(trainingData);
        double[] qerrors = sMap.qerrors(trainingData);
        double[] expected = {0.1296, 0.0759, 0.1187, 0.0971, 0.1143};

        assertTrue(MyMath.equalVector(expected, qerrors, 4));
    }

    public void testQuality() {
        System.out.println("quality");

        double[][] trainingData = {{0.7036, 0.1400},
                {0.4850, 0.5668},
                {0.1146, 0.8230},
                {0.6649, 0.6739},
                {0.3654, 0.9994}};
        SelfOrganizingMap sMap = new SelfOrganizingMap(trainingData);
        double qerror = sMap.quality(trainingData);

        assertEquals(0.107, MyMath.round(qerror, 3));
    }

    public void testHits() {
        System.out.println("hits");

        double[][] trainingData = {{0.7036, 0.1400},
                {0.4850, 0.5668},
                {0.1146, 0.8230},
                {0.6649, 0.6739},
                {0.3654, 0.9994}};
        SelfOrganizingMap sMap = new SelfOrganizingMap(trainingData);
        int[] hits = sMap.hits(trainingData);

        assertEquals(0, hits[0]);
        assertEquals(1, hits[11]);
    }

    public void testUmat() {
        System.out.println("umat");

        double[][] trainingData = {{0.7036, 0.1400},
                {0.4850, 0.5668},
                {0.1146, 0.8230},
                {0.6649, 0.6739},
                {0.3654, 0.9994}};
        SelfOrganizingMap sMap = new SelfOrganizingMap(trainingData);
        double[][] U;
        U = sMap.umat();
        double[][] expectedhexa = {{0.1028, 0.06, 0.1532},
                {0.1456, 0.2033, 0.1532},
                {0.1434, 0.0582, 0.1532},
                {0.1434, 0.1152, 0.1631},
                {0.1181, 0.056, 0.13},
                {0.1181, 0.13, 0.161},
                {0.13, 0.0678, 0.161},
                {0.2116, 0.2537, 0.2141},
                {0.1258, 0.0779, 0.1032},
                {0.1258, 0.0862, 0.1032},
                {0.0862, 0.0754, 0.0893}};

        assertTrue(MyMath.equalMatrix(expectedhexa, U, 4));

        sMap.getData().lattice = SelfOrganizingMap.Lattice.RECT;
        U = sMap.umat();
        double[][] expectedrect = {{0.1028, 0.06, 0.1066},
                {0.1456, 0.1075, 0.1532},
                {0.1434, 0.0582, 0.1532},
                {0.1434, 0.1111, 0.1631},
                {0.1181, 0.056, 0.161},
                {0.1181, 0.106, 0.161},
                {0.1181, 0.0678, 0.161},
                {0.2116, 0.1575, 0.2141},
                {0.1258, 0.0779, 0.1032},
                {0.1258, 0.0917, 0.1032},
                {0.1006, 0.0754, 0.0893}};

        assertTrue(MyMath.equalMatrix(expectedrect, U, 4));
    }

    public void testFromToString() {
        System.out.println("toString");

        double[][] trainingData = {{0.7036, 0.1400},
                {0.4850, 0.5668},
                {0.1146, 0.8230},
                {0.6649, 0.6739},
                {0.3654, 0.9994}};
        int[] classes = {0, 1, 1, 0, 1};

        SelfOrganizingMap sMap = new SelfOrganizingMap(trainingData);
        String content = sMap.toString();
        SelfOrganizingMap result = new SelfOrganizingMap(content);

        assertEquals(content, result.toString());
    }

}
