/*
 * SelfOrganizingMap.java
 *
 * Created on 2007-03-22, 16:15
 */

package iaik.som;

import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.InputStream;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOException;
import java.util.Arrays;

/**
 * This class is an implementation of Kohonen's Self-Organizing Map.
 * <br /><br />
 * It was inspired by SOM Toolbox for MATLAB:<br />
 * <a href="http://www.cis.hut.fi/projects/somtoolbox/">
 *    http://www.cis.hut.fi/projects/somtoolbox/</a>
 * <br /><br />
 * Limitations: <br />
 * mdim = 2 <br />
 * shape = 'sheet' <br />
 * no masks
 *
 * @author Gernot WALZL
 */
public class SelfOrganizingMap {

    /** for Matlab it has to be '\n' */
    protected static final String NEWLINE = "\n";
    protected SelfOrganizingMapData SOMData;

    /**
     * The lattice of the map can be hexagonal or rectangular.
     */
    public enum Lattice {
        HEXA, RECT;
    }

    /**
     * It holds possible neighborhood functions for updating the codebook.
     */
    public enum Neighborhood {
        GAUSSIAN, CUTGAUSS, EP, BUBBLE;
    }

    /**
     * The training phases can be rough, finetune or default.
     */
    public enum Training {
        DEFAULT, ROUGH, FINETUNE;
    }

    /**
     * Creates a new instance of SelfOrganizingMap
     * It initializes with default values.
     */
    public SelfOrganizingMap() {
        SOMData = new SelfOrganizingMapData();
        SOMData.lattice = Lattice.HEXA;
        SOMData.dimension = 2;
        SOMData.width = 5;
        SOMData.height = 5;
        SOMData.neighborhood = Neighborhood.GAUSSIAN;
        initialize();
    }

    /**
     * Creates a new instance of SelfOrganizingMap
     * It initializes with given values.
     */
    public SelfOrganizingMap(Lattice lattice,
            int dimension, int height, int width, Neighborhood neighborhood) {
        SOMData = new SelfOrganizingMapData();
        SOMData.lattice = lattice;
        SOMData.dimension = dimension;
        SOMData.height = height;
        SOMData.width = width;
        SOMData.neighborhood = neighborhood;
        initialize();
    }

    /**
     * Creates a new instance of SelfOrganizingMap
     * It takes the given data.
     */
    public SelfOrganizingMap(SelfOrganizingMapData SOMData) {
        this.SOMData = SOMData;
    }

    /**
     * Creates a new instance of SelfOragnizingMap
     * It trains an optimal codebook from the given training data.
     * <br />
     * som_make(D)
     */
    public SelfOrganizingMap(double[][] trainingData) {
        SOMData = new SelfOrganizingMapData();
        setup(trainingData);
        //randomizeCod();  // not usefull, because lininit writes codebook
        lininit(trainingData);
        TrainingSetupData sTrain;
        sTrain = generateTrainingSetup(trainingData.length, Training.ROUGH,
                Double.NaN);
        batchtrain(sTrain, trainingData);
        double prev_radius = sTrain.radius_fin;
        sTrain = generateTrainingSetup(trainingData.length, Training.FINETUNE,
                prev_radius);
        batchtrain(sTrain, trainingData);
    }

    /**
     * Creates a new instance of SelfOragnizingMap
     * It is the inversion of toString().
     */
    public SelfOrganizingMap(String content) {
        SOMData = new SelfOrganizingMapData();
        fromString(content);
    }

    /**
     * This function is internally used for initializaion.
     */
    protected void initialize() {
        SOMData.component_names = new String[SOMData.dimension];
        SOMData.codebook =
                new double[SOMData.width*SOMData.height][SOMData.dimension];
        for (int i = 0; i < SOMData.width*SOMData.height; i++)
            for (int j = 0; j < SOMData.dimension; j++)
                SOMData.codebook[i][j] = 0;
    }

    /**
     * This function sets the values of the SOM to the given training data.
     * <br />
     * som_topol_struct('data', D)
     */
    public void setup(double[][] trainingData) {
        int dlen = trainingData.length;
        int dim = trainingData[0].length;
        Lattice lattice = Lattice.HEXA;
        Neighborhood neighborhood = Neighborhood.GAUSSIAN;

        int munits = (int)Math.ceil(5.0 * Math.pow((double)dlen, 0.5));
        int[] msize = new int[2];

        if (dim == 1) {
            msize[0] = 1;
            msize[1] = munits;
        }
        if (dlen < 2) {
            msize[0] = (int)Math.round(Math.sqrt(munits));
            msize[1] = (int)Math.round(munits/msize[0]);
        }
        else {
            double[][] autocov = MyMath.autocovariance(trainingData);
            double[] eigenvalues = MyMath.eigenvalues(autocov);
            double ratio = 1.0;
            if (eigenvalues.length > 1) {
                Arrays.sort(eigenvalues);
                double[] max = new double[2];
                max[0] = eigenvalues[eigenvalues.length-1];
                max[1] = eigenvalues[eigenvalues.length-2];
                if (max[0] != 0 && max[1] * munits >= max[0])
                    ratio = Math.sqrt(max[0] / max[1]);
            }
            
            if (lattice == Lattice.HEXA) {
                msize[1] = (int)Math.round(
                        Math.sqrt((munits/ratio)*Math.sqrt(0.75)));
            }
            else {
                msize[1] = (int)Math.round(Math.sqrt(munits/ratio));
            }
            if (munits < msize[1])
                msize[1] = munits;
            msize[0] = (int)Math.round(munits / msize[1]);
        }

        SOMData.lattice = lattice;
        SOMData.dimension = dim;
        SOMData.height = msize[0];
        SOMData.width = msize[1];
        SOMData.neighborhood = neighborhood;

        initialize();
    }

    /**
     * This function randomizes the codebook.
     * <br />
     * som_map_struct(dim)
     */
    public void randomizeCod() {
        for (int i = 0; i < SOMData.width*SOMData.height; i++) {
            for (int j = 0; j < SOMData.dimension; j++)
                SOMData.codebook[i][j] = Math.random();
        }
    }

    /**
     * Initializes the codebook linearly
     * <br />
     * som_linint(D, sMap)
     */
    public void lininit(double[][] trainingData) {
        int dim = trainingData[0].length;
        double[][] A = MyMath.autocovariance(trainingData);
        double[] eigenvalues = MyMath.eigenvalues(A);
        double[][] eigenvectors = MyMath.eigenvectors(A);

        int[] maxind = new int[2];
        double[] eigvalcopy = MyMath.copyVector(eigenvalues);
        for (int i = 0; i < 2; i++) {
            maxind[i] = MyMath.maxIndex(eigvalcopy);
            eigvalcopy[maxind[i]] = -1 * Double.MAX_VALUE;
        }

        double[] mean = new double[dim];
        double[][] DTrans = MyMath.transpose(trainingData);
        for (int i = 0; i < dim; i++)
            mean[i] = MyMath.mean(DTrans[i]);

        // normalize eigenvectors and multiply them with sqrt(eigenvalues)
        double[][] VTrans = MyMath.transpose(eigenvectors);
        double[] normalized;
        for (int i = 0; i < dim; i++) {
            normalized = MyMath.normalize(VTrans[i]);
            for (int j = 0; j < normalized.length; j++)
                normalized[j] *= Math.sqrt(eigenvalues[i]);
            VTrans[i] = normalized;
        }

        int munits = SOMData.codebook.length;
        for (int i = 0; i < munits; i++) {
            for (int j = 0; j < dim; j++)
                SOMData.codebook[i][j] = mean[j];
        }

        double[][] coords = unitCoords(Lattice.RECT);
        double[][] coordsTrans = MyMath.transpose(coords);
        double[] temp = coordsTrans[0];
        coordsTrans[0] = coordsTrans[1];
        coordsTrans[1] = temp;

        double max = 0.0;
        for (int i = 0; i < 2; i++) {
            max = MyMath.max(coordsTrans[i]);
            if (max > 0) {
                for (int j = 0; j < coordsTrans[i].length; j++)
                    coordsTrans[i][j] /= max;
            }
            else {
                for (int j = 0; j < coordsTrans[i].length; j++)
                    coordsTrans[i][j] = 0.5;
            }
        }

        for (int i = 0; i < coordsTrans.length; i++) {
            for (int j = 0; j < coordsTrans[0].length; j++)
                coordsTrans[i][j] = (coordsTrans[i][j] - 0.5) * 2;
        }

        for (int n = 0; n < SOMData.codebook.length; n++) {
            for (int d = 0; d < 2; d++) {
                for (int i = 0; i < dim; i++) {
                    SOMData.codebook[n][i] = SOMData.codebook[n][i] +
                            (coordsTrans[d][n] * VTrans[maxind[d]][i]);
                }
            }
        }
    }

    /**
     * It creates a matrix of coordinates of each unit.
     * <br />
     * som_unit_coords(msize, lattice)
     */
    public double[][] unitCoords(Lattice lattice) {
        if (lattice == null)
            lattice = SOMData.lattice;
        int height = SOMData.height;
        int width = SOMData.width;
        int counter = 0;
        int munits = width * height;
        double[][] coords = new double[munits][2];

        for (int i = 0; i < width; i++) {
            for (int j = 0; j < height; j++) {
                coords[counter][0] = i;
                coords[counter][1] = j;
                counter++;
            }
        }

        if (lattice == Lattice.HEXA) {
            for (int i = 0; i < munits; i++) {
                if (i%2 == 1)
                    coords[i][0] += 0.5;
                coords[i][1] *= Math.sqrt(0.75);
            }
        }

        return coords;
    }

    /**
     * It creates a matrix of distances between two units.
     * <br />
     * som_unit_dists(sTopol)
     */
    public double[][] unitDists() {
        int munits = SOMData.codebook.length;
        double[][] Ud = new double[munits][munits];
        double[][] coords = unitCoords(SOMData.lattice);
        double[][] dco = new double[munits][2];

        // each step computes one line of Ud
        for (int i = 0; i < munits-1; i++) {
            // set unused to 0
            for (int j = 0; j <= i; j++) {
                for (int k = 0; k < 2; k++)
                    dco[j][k] = 0.0;
            }
            // only calculate right upper side of Ud
            for (int j = i+1; j < munits; j++) {
                for (int k = 0; k < 2; k++) {
                    dco[j][k] = Math.pow(coords[j][k] - coords[i][k], 2);
                }
                Ud[i][j] = Math.sqrt(MyMath.sum(dco[j]));
            }
        }

        // add transposed for left lower side of Ud
        double[][] UdTrans = MyMath.transpose(Ud);
        for (int i = 0; i < munits; i++) {
            for (int j = 0; j < munits; j++)
                Ud[i][j] += UdTrans[i][j];
        }

        return Ud;
    }

    /**
     * It generates a training setup for training. If the previous radius is
     * unknown, set it to Double.NaN.
     * <br />
     * som_train_struct(sMap, 'dlen', dlen, 'phase', phase)
     */
    public TrainingSetupData generateTrainingSetup(int dlen,
            Training training, double prev_radius) {
        int munits = SOMData.codebook.length;
        TrainingSetupData sTrain = new TrainingSetupData();

        sTrain.neighborhood = SOMData.neighborhood;

        if (Double.isNaN(prev_radius)) {
            int ms = SOMData.height;
            if (ms < SOMData.width) {
                ms = SOMData.width;
            }
            sTrain.radius_ini = Math.ceil(ms/8);
        }
        else {
            sTrain.radius_ini = prev_radius;
        }

        switch (training) {
            case ROUGH:
                sTrain.radius_fin = (int)sTrain.radius_ini/4;
                sTrain.trainlen = (int)Math.ceil(10*(double)munits/dlen);
                break;
            case FINETUNE:
                sTrain.trainlen = (int)Math.ceil(40*(double)munits/dlen);
                break;
            default:
                sTrain.trainlen = (int)Math.ceil(50*(double)munits/dlen);
                break;
        }
        if (sTrain.radius_ini < 1.0)
            sTrain.radius_ini = 1.0;
        if (sTrain.radius_fin < 1.0)
            sTrain.radius_fin = 1.0;
        if (sTrain.trainlen < 1)
            sTrain.trainlen = 1;

        return sTrain;
    }

    /**
     * This function calculates the distance for each training sample to each
     * unit on the codebook.
     * The distance between 2 vectors m and v can be transformed into a matrix
     * operation:
     * |m - v|^2 = sum_i ((m_i - v_i)^2)
     *           = sum_i (m_i^2 - 2*m_i*v_i + v_i^2)
     * Dist = (M.^2)*ones(dim,dlen)
     *        - 2*M*diag(ones(1,dim))*D'
     *        + ones(munits,dim)*(D'.^2)
     *      = (M.^2)*W1 - 2*M*WD*D' + W2*(D'.^2)
     *      = Dist1 + Dist2 + dist3
     */
    protected double[][] calculateDistances(double[][] sData) {
        double[][] D = MyMath.copyMatrix(sData);
        int munits = SOMData.codebook.length;
        int dim = SOMData.codebook[0].length;
        int dlen = D.length;
        double[][] Dist = new double[munits][dlen];
        double[][] M = SOMData.codebook;
        double[][] Msquared = new double[munits][dim];
        double[][] W1 = new double[dim][dlen];
        double[][] WD = new double[dim][dim];
        double[][] Dtrans;
        double[][] Dtranssquared = new double[dim][dlen];
        double[][] W2 = new double[munits][dim];
        double[][] Dist2;
        double[] dist3 = new double[dlen];

        // Msquared
        for (int i = 0; i < M.length; i++) {
            for (int j = 0; j < M[0].length; j++)
                Msquared[i][j] = Math.pow(M[i][j], 2);
        }

        // W1
        for (int i = 0; i < dim; i++) {
            for (int j = 0; j < dlen; j++) {
                if (D[j][i] == Double.NaN) {
                    W1[i][j] = 0.0;
                    D[j][i] = 0.0;
                }
                else {
                    W1[i][j] = 1.0;
                }
            }
        }

        // (M.^2)*ones(dim,dlen)
        Dist = MyMath.matmult(Msquared, W1);

        // -2*WD
        for (int i = 0; i < dim; i++) {
            for (int j = 0; j < dim; j++) {
                if (i == j)
                    WD[i][j] = -2.0;
                else
                    WD[i][j] = 0.0;
            }
        }

        Dtrans = MyMath.transpose(D);

        // 2*M*diag(ones(1,dim))*D'
        Dist2 = MyMath.matmult(M, WD);
        Dist2 = MyMath.matmult(Dist2, Dtrans);

        // W2
        for (int i = 0; i < munits; i++) {
            for (int j = 0; j < dim; j++)
                W2[i][j] = 1.0;
        }

        // Dtranssquared
        for (int i = 0; i < Dtrans.length; i++) {
            for (int j = 0; j < Dtrans[0].length; j++)
                Dtranssquared[i][j] = Math.pow(Dtrans[i][j], 2);
        }

        // ones(munits,dim)*(D'.^2)
        for (int i = 0; i < dlen; i++) {
            for (int j = 0; j < dim; j++)
                dist3[i] += Math.pow(D[i][j], 2);
        }

        for (int i = 0; i < munits; i++) {
            for (int j = 0; j < dlen; j++)
                Dist[i][j] += Dist2[i][j] + dist3[j];
        }

        return Dist;
    }

    /**
     * Batch training algorithm
     * <br />
     * som_batchtrain(sMap, D, sTrain)
     */
    public void batchtrain(TrainingSetupData sTrain, double[][] trainingData) {
        int munits = SOMData.codebook.length;
        int dlen = trainingData.length;
        int dim = trainingData[0].length;

        double[] radius = new double[sTrain.trainlen];
        for (int i = 0; i < sTrain.trainlen; i++) {
            radius[i] = sTrain.radius_ini
                    + (((sTrain.radius_fin - sTrain.radius_ini)
                    / (sTrain.trainlen - 1))
                    * i);
            radius[i] = Math.pow(radius[i], 2);
        }

        double[][] Ud = unitDists();
        for (int i = 0; i < Ud.length; i++) {
            for (int j = 0; j < Ud[0].length; j++)
                Ud[i][j] = Math.pow(Ud[i][j], 2);
        }

        double[][] known = new double[dlen][dim];
        for (int i = 0; i < dlen; i++) {
            for (int j = 0; j < dim; j++) {
                if (trainingData[i][j] == Double.NaN)
                    known[i][j] = 0.0;
                else
                    known[i][j] = 1.0;
            }
        }

        int[] bmus;

        double[][] P = new double[munits][dlen];
        double weight = 1.0;
        double[][] S;
        double[][] A;
        double[][] H = new double[munits][munits];

        for (int t = 0; t < sTrain.trainlen; t++) {
            bmus = bmus(trainingData);

            switch (sTrain.neighborhood) {
                case BUBBLE:
                    for (int i = 0; i < munits; i++) {
                        for (int j = 0; j < munits; j++) {
                            if (Ud[i][j] <= radius[t])
                                H[i][j] = 1.0;
                            else
                                H[i][j] = 0.0;
                        }
                    }
                    break;
                case GAUSSIAN:
                    for (int i = 0; i < munits; i++) {
                        for (int j = 0; j < munits; j++)
                            H[i][j] = Math.exp(-1*Ud[i][j]/(2*radius[t]));
                    }
                    break;
                case CUTGAUSS:
                    for (int i = 0; i < munits; i++) {
                        for (int j = 0; j < munits; j++) {
                            if (Ud[i][j] <= radius[t])
                                H[i][j] = Math.exp(-1*Ud[i][j]/(2*radius[t]));
                            else
                                H[i][j] = 0.0;
                        }
                    }
                    break;
                case EP:
                    for (int i = 0; i < munits; i++) {
                        for (int j = 0; j < munits; j++) {
                            if (Ud[i][j] <= radius[t])
                                H[i][j] = 1-(Ud[i][j]/radius[t]);
                            else
                                H[i][j] = 0.0;
                        }
                    }
                    break;
            }

            // update
            for (int i = 0; i < munits; i++) {
                for (int j = 0; j < dlen; j++) {
                    if (bmus[j] == i)
                        P[i][j] = weight;
                    else
                        P[i][j] = 0.0;
                }
            }

            S = MyMath.matmult(H, MyMath.matmult(P, trainingData));
            A = MyMath.matmult(H, MyMath.matmult(P, known));

            for (int i = 0; i < A.length; i++) {
                for (int j = 0; j < A[0].length; j++) {
                    if (A[i][j] > 0)
                        SOMData.codebook[i][j] = S[i][j] / A[i][j];
                }
            }
        }
    }

    /**
     * Best matching units
     * Set unknown elements to Double.NaN.
     * <br />
     * som_bmus(sMap, sData)
     */
    public int[] bmus(double[][] sData) {
        int dlen = sData.length;
        int[] bmus = new int[dlen];

        double[][] dist = calculateDistances(sData);
        double[][] distTrans = MyMath.transpose(dist);

        for (int i = 0; i < distTrans.length; i++)
            bmus[i] = MyMath.minIndex(distTrans[i]);

        return bmus;
    }

    /**
     * Quantization errors
     * Set unknown elements to Double.NaN.
     * <br />
     * [Bmus, Qerrors] = som_bmus(sMap, sData)
     */
    public double[] qerrors(double[][] sData) {
        int dlen = sData.length;
        double[] qerrors = new double[dlen];

        double[][] dist = calculateDistances(sData);
        double[][] distTrans = MyMath.transpose(dist);

        for (int i = 0; i < distTrans.length; i++)
            qerrors[i] = Math.sqrt(MyMath.min(distTrans[i]));

        return qerrors;
    }

    /**
     * Mean quantization error
     * <br />
     * som_quality(sMap, D)
     */
    public double quality(double[][] sData) {
        return MyMath.mean(qerrors(sData));
    }

    /**
     * Count hits per unit
     * <br />
     * som_hits(sMap, sData)
     */
    public int[] hits(double[][] sData) {
        int munits = SOMData.codebook.length;
        int[] hits = new int[munits];
        int[] bmus = bmus(sData);
        for (int i = 0; i < bmus.length; i++) {
           hits[bmus[i]]++;
        }
        return hits;
    }

    /**
     * Unified distance matrix
     * <br />
     * som_umat(sMap)
     */
    public double[][] umat() {
        int y = SOMData.height;
        int x = SOMData.width;
        int dim = SOMData.codebook[0].length;

        int uy = 2*y-1;
        int ux = 2*x-1;
        double[][] U = new double[uy][ux];
        double[][][] M = MyMath.reshape3d(SOMData.codebook, y, x, dim);

        // distances between map units
        double[] dx = new double[dim];
        double[] dy = new double[dim];
        double[] dz1 = new double[dim];
        double[] dz2 = new double[dim];

        switch (SOMData.lattice) {
            case RECT:
                for (int j = 0; j < y; j++) {
                    for (int i = 0; i < x; i++) {
                        // horizontal
                        if (i < (x-1)) {
                            for(int k = 0; k < dim; k++) {
                                dx[k] = Math.pow(M[j][i][k] - M[j][i+1][k], 2);
                            }
                            U[2*j][2*i+1] = Math.sqrt(MyMath.sum(dx));
                        }
                        // vertical
                        if (j < (y-1)) {
                            for(int k = 0; k < dim; k++) {
                                dy[k] = Math.pow(M[j][i][k] - M[j+1][i][k], 2);
                            }
                            U[2*j+1][2*i] = Math.sqrt(MyMath.sum(dy));
                        }
                        // diagonal
                        if ((i < (x-1)) && (j < (y-1))) {
                            for(int k = 0; k < dim; k++) {
                                dz1[k] = Math.pow(M[j][i][k]-M[j+1][i+1][k],2);
                                dz2[k] = Math.pow(M[j+1][i][k]-M[j][i+1][k],2);
                            }
                            U[2*j+1][2*i+1] = (Math.sqrt(MyMath.sum(dz1))
                                    + Math.sqrt(MyMath.sum(dz2)))
                                    / (2 * Math.sqrt(2));
                        }
                    }
                }
                break;
            case HEXA:
                for (int j = 0; j < y; j++) {
                    for (int i = 0; i < x; i++) {
                        // horizontal
                        if (i < (x-1)) {
                            for(int k = 0; k < dim; k++) {
                                dx[k] = Math.pow(M[j][i][k] - M[j][i+1][k], 2);
                            }
                            U[2*j][2*i+1] = Math.sqrt(MyMath.sum(dx));
                        }
                        // diagonal
                        if (j < (y-1)) {
                            for(int k = 0; k < dim; k++) {
                                dy[k] = Math.pow(M[j][i][k] - M[j+1][i][k], 2);
                            }
                            U[2*j+1][2*i] = Math.sqrt(MyMath.sum(dy));

                            if ((j%2 == 1) && (i < (x-1))) {
                                for(int k = 0; k < dim; k++) {
                                    dz1[k] = Math.pow(M[j][i][k]
                                            - M[j+1][i+1][k], 2);
                                }
                                U[2*j+1][2*i+1] = Math.sqrt(MyMath.sum(dz1));
                            }
                            else if ((j%2 == 0) && (i > 0)) {
                                for(int k = 0; k < dim; k++) {
                                    dz1[k] = Math.pow(M[j][i][k]
                                            - M[j+1][i-1][k], 2);
                                }
                                U[2*j+1][2*i-1] = Math.sqrt(MyMath.sum(dz1));
                            }
                        }
                    }
                }
                break;
        }

        // values on the units
        double[] a;
        if (uy == 1) {
            for (int i = 0; i < ux; i+=2) {
                if (i>0 && i<(ux-1)) {
                    a = new double[2];
                    a[0] = U[0][i-1];
                    a[1] = U[0][i+1];
                    U[0][i] = MyMath.median(a);
                }
                else if (i==0) {
                    U[0][i] = U[0][i+1];
                }
                else {
                    U[0][i] = U[0][i-1];
                }
            }
        }
        else if (ux == 1) {
            for (int j = 0; j < uy; j+=2) {
                if (j>0 && j<(uy-1)) {
                    a = new double[2];
                    a[0] = U[j-1][0];
                    a[1] = U[j+1][0];
                    U[j][0] = MyMath.median(a);
                }
                else if (j==0) {
                    U[j][0] = U[j+1][0];
                }
                else {
                    U[j][0] = U[j-1][0];
                }
            }
        }
        else if (SOMData.lattice == Lattice.RECT) {
            for (int j = 0; j < uy; j+=2) {
                for (int i = 0; i < ux; i+=2) {
                    // middle part of the map
                    if (i>0 && j>0 && i<(ux-1) && j<(uy-1)) {
                        a = new double[4];
                        a[0] = U[j][i-1];
                        a[1] = U[j][i+1];
                        a[2] = U[j-1][i];
                        a[3] = U[j+1][i];
                    }
                    // upper edge
                    else if (j==0 && i>0 && i<(ux-1)) {
                        a = new double[3];
                        a[0] = U[j][i-1];
                        a[1] = U[j][i+1];
                        a[2] = U[j+1][i];
                    }
                    // lower edge
                    else if (j==(uy-1) && i>0 && i<(ux-1)) {
                        a = new double[3];
                        a[0] = U[j][i-1];
                        a[1] = U[j][i+1];
                        a[2] = U[j-1][i];
                    }
                    // left edge
                    else if (i==0 && j>0 && j<(uy-1)) {
                        a = new double[3];
                        a[0] = U[j][i+1];
                        a[1] = U[j-1][i];
                        a[2] = U[j+1][i];
                    }
                    // right edge
                    else if (i==(ux-1) && j>0 && j<(uy-1)) {
                        a = new double[3];
                        a[0] = U[j][i-1];
                        a[1] = U[j-1][i];
                        a[2] = U[j+1][i];
                    }
                    // top left corner
                    else if (i==0 && j==0) {
                        a = new double[2];
                        a[0] = U[j][i+1];
                        a[1] = U[j+1][i];
                    }
                    // top right corner
                    else if (i==(ux-1) && j==0) {
                        a = new double[2];
                        a[0] = U[j][i-1];
                        a[1] = U[j+1][i];
                    }
                    // bottom left corner
                    else if (i==0 && j==(uy-1)) {
                        a = new double[2];
                        a[0] = U[j][i+1];
                        a[1] = U[j-1][i];
                    }
                    // bottom right corner
                    else if (i==(ux-1) && j==(uy-1)) {
                        a = new double[2];
                        a[0] = U[j][i-1];
                        a[1] = U[j-1][i];
                    }
                    else {
                        a = new double[1];
                        a[0] = 0;
                    }
                    U[j][i] = MyMath.median(a);
                }
            }
        }
        else if (SOMData.lattice == Lattice.HEXA) {
            for (int j = 0; j < uy; j+=2) {
                for (int i = 0; i < ux; i+=2) {
                    // middle part of the map
                    if (i>0 && j>0 && i<(ux-1) && j<(uy-1)) {
                        a = new double[6];
                        a[0] = U[j][i-1];
                        a[1] = U[j][i+1];
                        if (j%4 == 0) {
                            a[2] = U[j-1][i-1];
                            a[3] = U[j-1][i];
                            a[4] = U[j+1][i-1];
                            a[5] = U[j+1][i];
                        }
                        else {
                            a[2] = U[j-1][i];
                            a[3] = U[j-1][i+1];
                            a[4] = U[j+1][i];
                            a[5] = U[j+1][i+1];
                        }
                    }
                    // upper edge
                    else if (j==0 && i>0 && i<(ux-1)) {
                        a = new double[4];
                        a[0] = U[j][i-1];
                        a[1] = U[j][i+1];
                        a[2] = U[j+1][i-1];
                        a[3] = U[j+1][i];
                    }
                    // lower edge
                    else if (j==(uy-1) && i>0 && i<(ux-1)) {
                        a = new double[4];
                        a[0] = U[j][i-1];
                        a[1] = U[j][i+1];
                        if (j%4 == 0) {
                            a[2] = U[j-1][i-1];
                            a[3] = U[j-1][i];
                        }
                        else {
                            a[2] = U[j-1][i];
                            a[3] = U[j-1][i+1];
                        }
                    }
                    // left edge
                    else if (i==0 && j>0 && j<(uy-1)) {
                        if (j%4 == 0) {
                            a = new double[3];
                            a[0] = U[j][i+1];
                            a[1] = U[j-1][i];
                            a[2] = U[j+1][i];
                        }
                        else {
                            a = new double[5];
                            a[0] = U[j][i+1];
                            a[1] = U[j-1][i];
                            a[2] = U[j-1][i+1];
                            a[3] = U[j+1][i];
                            a[4] = U[j+1][i+1];
                        }
                    }
                    // right edge
                    else if (i==(ux-1) && j>0 && j<(uy-1)) {
                        if (j%4 == 0) {
                            a = new double[5];
                            a[0] = U[j][i-1];
                            a[1] = U[j-1][i];
                            a[2] = U[j-1][i-1];
                            a[3] = U[j+1][i];
                            a[4] = U[j+1][i-1];
                        }
                        else {
                            a = new double[3];
                            a[0] = U[j][i-1];
                            a[1] = U[j-1][i];
                            a[2] = U[j+1][i];
                        }
                    }
                    // top left corner
                    else if (i==0 && j==0) {
                        a = new double[2];
                        a[0] = U[j][i+1];
                        a[1] = U[j+1][i];
                    }
                    // top right corner
                    else if (i==(ux-1) && j==0) {
                        a = new double[3];
                        a[0] = U[j][i-1];
                        a[1] = U[j+1][i-1];
                        a[2] = U[j+1][i];
                    }
                    // bottom left corner
                    else if (i==0 && j==(uy-1)) {
                        if (j%4 == 0) {
                            a = new double[2];
                            a[0] = U[j][i+1];
                            a[1] = U[j+1][i];
                        }
                        else {
                            a = new double[3];
                            a[0] = U[j][i+1];
                            a[1] = U[j-1][i];
                            a[2] = U[j-1][i+1];
                        }
                    }
                    // bottom right corner
                    else if (i==(ux-1) && j==(uy-1)) {
                        if (j%4 == 0) {
                            a = new double[3];
                            a[0] = U[j][i-1];
                            a[1] = U[j-1][i];
                            a[2] = U[j-1][i-1];
                        }
                        else {
                            a = new double[2];
                            a[0] = U[j][i-1];
                            a[1] = U[j-1][i];
                        }
                    }
                    else {
                        a = new double[1];
                        a[0] = 0;
                    }
                    U[j][i] = MyMath.median(a);
                }
            }
        }

        return U;
    }


    public void setData(SelfOrganizingMapData SOMData) {
        this.SOMData = SOMData;
    }


    public SelfOrganizingMapData getData() {
        return this.SOMData;
    }


    public String toString() {
        StringBuffer result = new StringBuffer();
        String line = new String();

        line += SOMData.dimension + " ";
        switch (SOMData.lattice) {
            case HEXA: line += "hexa"; break;
            case RECT: line += "rect"; break;
        }
        line += " " + SOMData.width + " " + SOMData.height + " ";
        switch (SOMData.neighborhood) {
            case GAUSSIAN: line += "gaussian"; break;
            case CUTGAUSS: line += "cutgauss"; break;
            case EP: line += "ep"; break;
            case BUBBLE: line += "bubble"; break;
        }
        result.append(line + NEWLINE);

        line = "#n ";
        for (int i = 0; i < SOMData.dimension; i++)
            line += SOMData.component_names[i] + " ";
        result.append(line + NEWLINE);

        for (int i = 0; i < SOMData.width*SOMData.height; i++) {
            line = "";
            for (int j = 0; j < SOMData.dimension; j++)
                line += MyMath.round(SOMData.codebook[i][j], 6) + " ";
            result.append(line + NEWLINE);
        }

        return result.toString();
    }


    private void fromString(String content) {
        String[] line = content.split(NEWLINE);
        String[] word = line[0].split(" ");

        SOMData.dimension = Integer.valueOf(word[0]).intValue();

        if (word[1].equals("hexa"))
            SOMData.lattice = Lattice.HEXA;
        else if (word[1].equals("rect"))
            SOMData.lattice = Lattice.RECT;

        SOMData.width = Integer.valueOf(word[2]).intValue();
        SOMData.height = Integer.valueOf(word[3]).intValue();

        if (word[4].equals("gaussian"))
            SOMData.neighborhood = Neighborhood.GAUSSIAN;
        else if (word[4].equals("cutgauss"))
            SOMData.neighborhood = Neighborhood.CUTGAUSS;
        else if (word[4].equals("ep"))
            SOMData.neighborhood = Neighborhood.EP;
        else if (word[4].equals("bubble"))
            SOMData.neighborhood = Neighborhood.BUBBLE;

        initialize();

        word = line[1].split(" ");
        for (int i = 0; i < SOMData.dimension; i++)
            SOMData.component_names[i] = word[i+1];

        for (int i = 0; i < SOMData.width*SOMData.height; i++) {
            word = line[i+2].split(" ");
            for (int j = 0; j < SOMData.dimension; j++)
                SOMData.codebook[i][j] = Float.valueOf(word[j]);
        }
    }

    /**
     * Exports to MATLAB SOM Toolbox som_read_cod
     * <br />
     * som_write_cod(sMap, filename)
     */
    public void writeCod(OutputStream out) {
        PrintWriter pw = new PrintWriter(out);
        pw.print(this.toString());
        pw.close();
    }

    /**
     * Imports from MATLAB SOM Toolbox som_write_cod
     * <br />
     * som_read_cod(filename)
     */
    public void readCod(InputStream in) {
        BufferedReader br = new BufferedReader(new InputStreamReader(in));
        String line;
        String content = "";
        try {
            while ((line = br.readLine()) != null)
                content += line + NEWLINE;
            br.close();
            fromString(content);
        } catch(IOException ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
    }

}
