/*
 * MyMath.java
 *
 * Created on 2007-04-03, 16:25
 */

package iaik.som;

import java.util.Arrays;
import Jama.Matrix;
import Jama.EigenvalueDecomposition;
import java.util.Iterator;
import java.util.LinkedList;

/**
 * This class holds basic static methods for vectors and matrices.
 *
 * @author Gernot WALZL
 */
public class MyMath {

    /**
     * The autocovariance matrix is defined as:<br />
     * gammaXX[n,m] = E( (X[n] - E(X[n])) * (X[m] - E(X[m])) )
     * <br /><br />
     * It is based on the standard covariance matrix:<br />
     * gammaXY[n,m] = E( (X[n] - E(X[n])) * (Y[m] - E(Y[m])) )
     *
     * @param matrix   X
     * @return         gammaXX
     */
    public static double[][] autocovariance(double[][] matrix) {
        int dlen = matrix.length;
        int dim = matrix[0].length;
        double[][] A = new double[dim][dim];
        double[][] copy = copyMatrix(matrix);

        // subtract mean-values for each dimension
        double mean = 0.0;
        for (int current_dim = 0; current_dim < dim; current_dim++) {
            mean = 0.0;
            for (int i = 0; i < dlen; i++)
                mean += copy[i][current_dim];
            mean = mean / dlen;
            for (int i = 0; i < dlen; i++)
                copy[i][current_dim] -= mean;
        }

        // calculate covariance matrix
        double[] c = new double[dlen];
        double csum = 0.0;
        for (int i = 0; i < dim; i++) {
            for (int j = i; j < dim; j++) {
                csum = 0.0;
                for (int current = 0; current < dlen; current++) {
                    c[current] = copy[current][i] * copy[current][j];
                    csum += c[current];
                }
                A[i][j] = csum / c.length;
                A[j][i] = A[i][j];
            }
        }

        return A;
    }

    /**
     * Eigenvalues s are solutions for the following formula:<br />
     * det(A-sI) = 0
     *
     * @param squarematrix   A, size n * n
     * @return               s, only real-valued
     */
    public static double[] eigenvalues(double[][] squarematrix) {
        Matrix jamaMatrix = new Matrix(squarematrix);
        EigenvalueDecomposition eig = new EigenvalueDecomposition(jamaMatrix);
        double[] eigenvalues = eig.getRealEigenvalues();
        return eigenvalues;
    }

    /**
     * Eigenvectors v solve this equation:<br />
     * (A-sI)v = 0
     *
     * @param squarematrix   A, size n * n
     * @return               v
     */
    public static double[][] eigenvectors(double[][] squarematrix) {
        Matrix jamaMatrix = new Matrix(squarematrix);
        EigenvalueDecomposition eig = new EigenvalueDecomposition(jamaMatrix);
        double[][] eigenvectors = eig.getV().getArray();
        return eigenvectors;
    }

    /**
     * A normalization divides every item of the vector by the euclidean norm
     * of the whole vector.
     *
     * @param vector   input
     * @return         normalized vector
     */
    public static double[] normalize(double[] vector) {
        int length = vector.length;
        double[] returned = copyVector(vector);

        // calculating norm
        double norm = 0.0;
        for (int i = 0; i < length; i++) {
            norm += Math.pow(returned[i], 2);
        }
        norm = Math.sqrt(norm);

        // normalize elements
        for (int i = 0; i < length; i++) {
            returned[i] /= norm;
        }

        return returned;
    }

    /**
     * The transpose of a matrix holds the same values, but with differend
     * indexes. The columns became rows and the rows changed to columns.
     *
     * @param matrix   matrix A
     * @return         transpose of A
     */
    public static double[][] transpose(double[][] matrix) {
        int oldheight = matrix.length;
        int oldwidth = matrix[0].length;
        double[][] transpose = new double[oldwidth][oldheight];

        for (int i = 0; i < oldheight; i++) {
            for (int j = 0; j < oldwidth; j++)
                transpose[j][i] = matrix[i][j];
        }

        return transpose;
    }

    /**
     * It sums up all items of the vector and divides it by the number of all
     * elements.
     *
     * @param vector   input
     * @return         mean value
     */
    public static double mean(double[] vector) {
        int length = vector.length;
        double mean = 0.0;

        for (int i = 0; i < length; i++) {
            mean += vector[i];
        }
        mean /= length;

        return mean;
    }

    /**
     * It sorts the array and returns the value from the middle.
     *
     * @param vector   input
     * @return         median value
     */
    public static double median(double[] vector) {
        double median = 0.0;
        int lengthz = vector.length - 1;
        double[] copy = copyVector(vector);
        Arrays.sort(copy);
        if ((lengthz % 2) == 0)
            median = copy[lengthz/2];
        else
            median = (copy[(int)lengthz/2] + copy[(int)(lengthz/2)+1]) / 2;

        return median;
    }

    /**
     * This function searches for the highest value.
     *
     * @param vector   input
     * @return         maximum
     */
    public static double max(double[] vector) {
        double max = vector[0];
        for (int i = 1; i < vector.length; i++) {
            if (max < vector[i])
                max = vector[i];
        }
        return max;
    }

    public static int max(int[] vector) {
        int max = vector[0];
        for (int i = 1; i < vector.length; i++) {
            if (max < vector[i])
                max = vector[i];
        }
        return max;
    }

    /**
     * This function searches for the lowest value.
     *
     * @param vector   input
     * @return         minimum
     */
    public static double min(double[] vector) {
        double min = vector[0];
        for (int i = 1; i < vector.length; i++) {
            if (min > vector[i])
                min = vector[i];
        }
        return min;
    }

    /**
     * This function searches for the index of the highest value.
     *
     * @param vector   input
     * @return         index of maximum
     */
    public static int maxIndex(double[] vector) {
        int index = 0;
        double max = vector[0];
        for (int i = 1; i < vector.length; i++) {
            if (max < vector[i]) {
                max = vector[i];
                index = i;
            }
        }
        return index;
    }

    /**
     * This function searches for the index of the lowest value.
     *
     * @param vector   input
     * @return         index of minimum
     */
    public static int minIndex(double[] vector) {
        int index = 0;
        double min = vector[0];
        for (int i = 1; i < vector.length; i++) {
            if (min > vector[i]) {
                min = vector[i];
                index = i;
            }
        }
        return index;
    }

    /**
     * It sums up all elements of the given vector.
     *
     * @param vector   input
     * @return         sum
     */
    public static double sum(double[] vector) {
        double sum = 0.0;

        for (int i = 0; i < vector.length; i++)
            sum += vector[i];

        return sum;
    }

    public static int sum(int[] vector) {
        int sum = 0;

        for (int i = 0; i < vector.length; i++)
            sum += vector[i];

        return sum;
    }


    /**
     * A matrix multiplication sums up the rows of matrix A multiplicated with
     * the columns of matrix B. It is not commutative.
     *
     * @param matrixA   A
     * @param matrixB   B
     * @return          A*B
     */
    public static double[][] matmult(double[][] matrixA, double[][] matrixB) {
        int height = matrixA.length;
        int width = matrixB[0].length;
        double[][] result = new double[height][width];
        double value;

        for (int i = 0; i < height; i++) {
            for (int j = 0; j < width; j++) {
                value = 0.0;
                for (int k = 0; k < matrixA[0].length; k++) {
                    value += matrixA[i][k] * matrixB[k][j];
                }
                result[i][j] = value;
            }
        }

        return result;
    }

    /**
     * This function shapes a 2d matrix to a 3d matrix. The number of elements
     * have to fit.
     *
     * @param matrix   2d input matrix
     * @param height   height of the new 3d matrix
     * @param width    width of the new 3d matrix
     * @param depth    depth of the new 3d matrix
     * @return         3d output matrix
     */
    public static double[][][] reshape3d(double[][] matrix,
            int height, int width, int depth) {
        double[][][] result = new double[height][width][depth];

        // check if elements fit
        int oldheight = matrix.length;
        int oldwidth = matrix[0].length;
        if (oldheight * oldwidth != height * width * depth)
            return null;

        // create a list of elements of the matrix
        double[] elements = new double[oldheight*oldwidth];
        int counter = 0;
        for (int x = 0; x < oldwidth; x++) {
            for (int y = 0; y < oldheight; y++) {
                elements[counter] = matrix[y][x];
                counter++;
            }
        }

        // fill these elements into the new matrix
        counter = 0;
        for (int z = 0; z < depth; z++) {
            for (int x = 0; x < width; x++) {
                for (int y = 0; y < height; y++) {
                    result[y][x][z] = elements[counter];
                    counter++;
                }
            }
        }

        return result;
    }

    /**
     * It generates a copy of a matrix.
     *
     * @param src   source
     * @return      destination
     */
    public static double[][] copyMatrix(double[][] src) {
        int height = src.length;
        int width = src[0].length;
        double[][] dst = new double[height][width];

        for (int i = 0; i < height; i++) {
            for (int j = 0; j < width; j++)
                dst[i][j] = src[i][j];
        }

        return dst;
    }

    /**
     * It generates a copy of a vector.
     *
     * @param src   source
     * @return      destination
     */
    public static double[] copyVector(double[] src) {
        int length = src.length;
        double[] dst = new double[length];

        for (int i = 0; i < length; i++)
            dst[i] = src[i];

        return dst;
    }

    /**
     * This function rounds any given number.
     *
     * @param number     input
     * @param decimals   number of decimals
     * @return           rounded number
     */
    public static double round(double number, int decimals) {
        int i = (int)Math.pow(10, decimals);
        return (double)Math.round(number * i)/i;
    }

    /**
     * It checks if two matrices are equal.
     *
     * @param matrixA    input
     * @param matrixB    input
     * @param decimals   number of decimals to round
     * @return           equal or not
     */
    public static boolean equalMatrix(double[][] matrixA, double[][] matrixB,
            int decimals) {

        // check dimensions
        int height = matrixA.length;
        int width = matrixA[0].length;

        if (matrixA.length != matrixB.length)
            return false;

        for (int i = 0; i < height; i++) {
            if (matrixA[i].length != matrixB[i].length)
                return false;
        }

        // check content
        for (int i = 0; i < height; i++) {
            for (int j = 0; j < width; j++) {
                if (round(matrixA[i][j], decimals) !=
                        round(matrixB[i][j], decimals))
                    return false;
            }
        }

        return true;
    }

    public static boolean equalMatrix(int[][] matrixA, int[][] matrixB) {

        // check dimensions
        int height = matrixA.length;
        int width = matrixA[0].length;

        if (matrixA.length != matrixB.length)
            return false;

        for (int i = 0; i < height; i++) {
            if (matrixA[i].length != matrixB[i].length)
                return false;
        }

        // check content
        for (int i = 0; i < height; i++) {
            for (int j = 0; j < width; j++) {
                if (matrixA[i][j] != matrixB[i][j])
                    return false;
            }
        }

        return true;
    }

    public static boolean equalMatrix(boolean[][] matrixA,
            boolean[][] matrixB) {

        // check dimensions
        int height = matrixA.length;
        int width = matrixA[0].length;

        if (matrixA.length != matrixB.length)
            return false;

        for (int i = 0; i < height; i++) {
            if (matrixA[i].length != matrixB[i].length)
                return false;
        }

        // check content
        for (int i = 0; i < height; i++) {
            for (int j = 0; j < width; j++) {
                if (matrixA[i][j] != matrixB[i][j])
                    return false;
            }
        }

        return true;
    }

    /**
     * It checks if two vectors are equal.
     *
     * @param vectorA    input
     * @param vectorB    input
     * @param decimals   number of decimals to round
     * @return           equal or not
     */
    public static boolean equalVector(double[] vectorA, double[] vectorB,
            int decimals) {

        // check dimensions
        int length = vectorA.length;

        if (vectorA.length != vectorB.length)
            return false;

        // check content
        for (int i = 0; i < length; i++) {
            if (round(vectorA[i], decimals) != round(vectorB[i], decimals))
                return false;
        }

        return true;
    }

    public static boolean equalVector(int[] vectorA, int[] vectorB) {

        // check dimensions
        int length = vectorA.length;

        if (vectorA.length != vectorB.length)
            return false;

        // check content
        for (int i = 0; i < length; i++) {
            if (vectorA[i] != vectorB[i])
                return false;
        }

        return true;
    }


    public static String getStringFromMatrix(double[][] matrix, int decimals) {
        int height = matrix.length;
        int width = matrix[0].length;
        String newline = System.getProperty("line.separator");
        StringBuffer result = new StringBuffer();
        result.append("[");
        for (int i = 0; i < height; i++) {
            result.append(getStringFromVector(matrix[i], decimals));
            //result.append("[");
            //for (int j = 0; j < width; j++) {
            //    result.append(round(matrix[i][j], decimals));
            //    if (j < (width-1))
            //        result.append(", ");
            //}
            //result.append("]");
            if (i < (height-1)) {
                result.append(";");
                result.append(newline);
            }
        }
        result.append("]");
        result.append(newline);
        return result.toString();
    }


    public static String getStringFromVector(double[] vector, int decimals) {
        int length = vector.length;
        StringBuffer result = new StringBuffer();
        result.append("[");
        for (int i = 0; i < length; i++) {
            result.append(round(vector[i], decimals));
            if (i < (length-1))
                result.append(", ");
        }
        result.append("]");
        return result.toString();
    }

    public static String getStringFromVector(int[] vector) {
        int length = vector.length;
        StringBuffer result = new StringBuffer();
        result.append("[");
        for (int i = 0; i < length; i++) {
            result.append(vector[i]);
            if (i < (length-1))
                result.append(", ");
        }
        result.append("]");
        return result.toString();
    }


    public static double[] toVector(final String strVec) {
        String noBraces = strVec.replaceAll("[\\[\\]]", "");
        String noSpaces = noBraces.trim();
        noSpaces = noSpaces.replaceAll("\\s*:\\s*", ":");
        noSpaces = noSpaces.replaceAll("\\s*,\\s*|\\s+", ",");
        String[] numbers = noSpaces.split(",");
        String[] splitedNumber = null;
        LinkedList<Double> values = new LinkedList<Double>();
        for (int i = 0; i < numbers.length; i++) {
            splitedNumber = numbers[i].split(":");
            try {
                if (splitedNumber.length == 1) {
                    values.add(Double.valueOf(numbers[i]));
                } else if (splitedNumber.length == 2) {
                    for (Double value = Double.valueOf(splitedNumber[0]);
                    value <= Double.valueOf(splitedNumber[1]);
                    value += 1.0)
                        values.add(value);
                } else if (splitedNumber.length == 3) {
                    for (Double value = Double.valueOf(splitedNumber[0]);
                    value <= Double.valueOf(splitedNumber[2]);
                    value += Double.valueOf(splitedNumber[1]))
                        values.add(value);
                } else {
                    values.add(null);
                }
            } catch(Exception ex) {
                values.add(null);
            }
        }

        double[] result = new double[values.size()];
        Double value = null;
        Iterator<Double> it = values.iterator();
        for (int i = 0; i < result.length; i++) {
            value = it.next();
            if (value == null)
                result[i] = 0.0;
            else
                result[i] = value.doubleValue();
        }
        return result;
    }


    public static double[][] toMatrix(final String strMat) {
        String noBraces = strMat.replaceAll("[\\[\\]]", "");
        String[] vectors = noBraces.split("\\s*;\\s*");
        int length = vectors.length;
        double[][] result = new double[length][];
        for (int i = 0; i < length; i++) {
            result[i] = toVector(vectors[i]);
        }
        return result;
    }


    public static int[] toIntArr(final double[] vector) {
        int length = vector.length;
        int[] result = new int[length];
        for (int i = 0; i < length; i++) {
            result[i] = Double.valueOf(vector[i]).intValue();
        }
        return result;
    }

}
