import math

__author__="Gernot WALZL"
__date__ ="2009-11-05"

class Looker:
    """
    z-up coordinate system
    """

    def __init__(self):
        self.eye = [1.5, 1.5, 1.5]      # current position
        self.center = [0.0, 0.0, 0.0]   # look at
        self.up = [0.0, 0.0, 1.0]       # up vector
        self.move_speed = 0.05
        self.look_speed = 0.02

    def __str__(self):
        result = 'eye=<' + ','.join(str(v) for v in self.eye) + '>\n'
        result += 'center=<' + ','.join(str(v) for v in self.center) + '>\n'
        result += 'up=<' + ','.join(str(v) for v in self.up) + '>\n'
        return result

    def reset(self):
        self.eye = [1.5, 1.5, 1.5]
        self.center = [0.0, 0.0, 0.0]
        self.up = [0.0, 0.0, 1.0]

    def angle_xy(self):
        delta = [0.0, 0.0]
        for i in range(0, len(delta)):
            delta[i] = self.center[i] - self.eye[i]
        return math.atan2(delta[1], delta[0])

    def angle_z(self):
        delta = [0.0, 0.0, 0.0]
        for i in range(0, len(delta)):
            delta[i] = self.center[i] - self.eye[i]
        length = math.sqrt(delta[0]**2 + delta[1]**2)
        return math.atan2(delta[2], length)

    def move(self, delta=[0.0, 0.0, 0.0]):
        for i in range(0, len(delta)):
            self.eye[i] += delta[i]
            self.center[i] += delta[i]

    def move_fb(self, amount=0.0):
        angle_xy = self.angle_xy()
        angle_z = self.angle_z()
        delta = [0.0, 0.0, 0.0]
        delta[0] = amount * math.cos(angle_xy) * math.cos(angle_z)
        delta[1] = amount * math.sin(angle_xy) * math.cos(angle_z)
        delta[2] = amount * math.sin(angle_z)
        self.move(delta)

    def move_ud(self, amount=0.0):
        delta = [0.0, 0.0, amount]
        self.move(delta)

    def strafe_lr(self, amount=0.0):
        angle_xy = self.angle_xy()
        angle_xy -= math.pi/2.0
        delta = [0.0, 0.0, 0.0]
        delta[0] = amount * math.cos(angle_xy)
        delta[1] = amount * math.sin(angle_xy)
        self.move(delta)

    def strafe_ud(self, amount=0.0):
        angle_xy = self.angle_xy()
        angle_z = self.angle_z()
        delta = [0.0, 0.0, 0.0]
        delta[0] = -amount * math.cos(angle_xy) * math.sin(angle_z)
        delta[1] = -amount * math.sin(angle_xy) * math.sin(angle_z)
        delta[2] = amount * math.cos(angle_z)
        self.move(delta)

    def forward(self):
        self.move_fb(self.move_speed)

    def backpedal(self):
        self.move_fb(-self.move_speed)

    def move_up(self):
        self.move_ud(self.move_speed)

    def move_down(self):
        self.move_ud(-self.move_speed)

    def strafe_left(self):
        self.strafe_lr(-self.move_speed)

    def strafe_right(self):
        self.strafe_lr(self.move_speed)

    def strafe_up(self):
        self.strafe_ud(self.move_speed)

    def strafe_down(self):
        self.strafe_ud(-self.move_speed)

    def look_lr(self, amount=0.0):
        angle_xy = self.angle_xy()
        angle_xy -= amount
        old_delta = [0.0, 0.0]
        for i in range(0, len(old_delta)):
            old_delta[i] = self.center[i] - self.eye[i]
        length = math.sqrt(old_delta[0]**2 + old_delta[1]**2)
        new_delta = [0.0, 0.0]
        new_delta[0] = length * math.cos(angle_xy)
        new_delta[1] = length * math.sin(angle_xy)
        for i in range(0, len(new_delta)):
            self.center[i] = self.eye[i] + new_delta[i]

    def look_ud(self, amount=0.0):
        angle_z = self.angle_z()
        angle_z += amount
        if (-math.pi/2.0 >= angle_z or angle_z >= math.pi/2.0):
            return
        angle_xy = self.angle_xy()
        old_delta = [0.0, 0.0, 0.0]
        length = 0.0
        for i in range(0, len(old_delta)):
            old_delta[i] = self.center[i] - self.eye[i]
            length += old_delta[i]**2
        length = math.sqrt(length)
        new_length_xy = length * math.cos(angle_z)
        new_delta = [0.0, 0.0, 0.0]
        new_delta[0] = new_length_xy * math.cos(angle_xy)
        new_delta[1] = new_length_xy * math.sin(angle_xy)
        new_delta[2] = length * math.sin(angle_z)
        for i in range(0, len(new_delta)):
            self.center[i] = self.eye[i] + new_delta[i]

    def look_left(self):
        self.look_lr(-self.look_speed)

    def look_right(self):
        self.look_lr(self.look_speed)

    def look_up(self):
        self.look_ud(self.look_speed)

    def look_down(self):
        self.look_ud(-self.look_speed)

    def rotate_world_lr(self, amount=0.0):
        angle = math.atan2(self.eye[1], self.eye[0])
        length = math.sqrt(self.eye[0]**2 + self.eye[1]**2)
        angle -= amount
        self.eye[0] = length * math.cos(angle)
        self.eye[1] = length * math.sin(angle)
        angle = math.atan2(self.center[1], self.center[0])
        length = math.sqrt(self.center[0]**2 + self.center[1]**2)
        angle -= amount
        self.center[0] = length * math.cos(angle)
        self.center[1] = length * math.sin(angle)

    def rotate_world_ud(self, amount=0.0):
        angle_z = self.angle_z()
        angle_z += 2*amount
        if (-math.pi/2.0 >= angle_z or angle_z >= math.pi/2.0):
            return
        angle_xy = math.atan2(self.eye[1], self.eye[0])

        # rotate x axis
        factor = math.sin(angle_xy)

        angle = math.atan2(self.eye[2], self.eye[1])
        length = math.sqrt(self.eye[2]**2 + self.eye[1]**2)
        angle -= amount * factor
        self.eye[1] = length * math.cos(angle)
        self.eye[2] = length * math.sin(angle)

        angle = math.atan2(self.center[2], self.center[1])
        length = math.sqrt(self.center[2]**2 + self.center[1]**2)
        angle -= amount * factor
        self.center[1] = length * math.cos(angle)
        self.center[2] = length * math.sin(angle)

        # rotate y axis
        factor = math.cos(angle_xy)

        angle = math.atan2(self.eye[2], self.eye[0])
        length = math.sqrt(self.eye[2]**2 + self.eye[0]**2)
        angle -= amount * factor
        self.eye[0] = length * math.cos(angle)
        self.eye[2] = length * math.sin(angle)

        angle = math.atan2(self.center[2], self.center[0])
        length = math.sqrt(self.center[2]**2 + self.center[0]**2)
        angle -= amount * factor
        self.center[0] = length * math.cos(angle)
        self.center[2] = length * math.sin(angle)

