import time
import random
import cProfile
import pstats
import os
import sys
import types
from hcs.data.AdjacencyListMesh import Node
from hcs.algo.common import count_types
from hcs.algo.QuickHull import quick_hull


__author__="Gernot WALZL"
__date__ ="2010-02-24"


def random_create_point():
    result = [0.0, 0.0, 0.0]
    for i in range(0, 3):
        result[i] = (random.random()-0.5)*2.0
    return result


def random_create_nodes(mesh, num_points):
    random.seed(time.time())
    i = 0
    while (i < num_points):
        node = Node(random_create_point())
        mesh.add_node(node)
        i += 1


def random_set_type(mesh, num_types=2):
    if (num_types == 0):
        for node in mesh.nodes:
            node.type = 0
    elif (num_types == 1):
        for node in mesh.nodes:
            node.type = 1
    else:
        random.seed(time.time())
        for node in mesh.nodes:
            node.type = random.randint(1, num_types)


def random_mesh_for_quick_hull(mesh):
    mesh.clear()
    num_points = 16
    print('Creating '+str(num_points)+' random points')
    random_create_nodes(mesh, num_points)


def random_mesh_for_split_hull(mesh):
    mesh.clear()
    num_points = 16
    print('Creating '+str(num_points)+' random points')
    random_create_nodes(mesh, num_points)
    print('Computing convex hull')
    quick_hull(mesh)
    print('Removing points, which are not on hull')
    mesh.remove_nodes_not_connected()
    print('Randomized setting type of nodes')
    random_set_type(mesh)


def random_mesh_for_rand_multi_split(mesh):
    mesh.clear()
    num_points = 32
    print('Creating '+str(num_points)+' random points')
    random_create_nodes(mesh, 16)
    print('Computing convex hull')
    quick_hull(mesh)
    print('Removing points, which are not on hull')
    mesh.remove_nodes_not_connected()
    print('Randomized setting type of nodes')
    random_set_type(mesh, 3)


def run_quick_hull_demo(mesh, animator=None):
    if ((len(mesh.nodes) == 0) or (len(mesh.edges) != 0)):
        random_mesh_for_quick_hull(mesh)
    run_demo_analyze('quick_hull(mesh, animator)', mesh, animator)


def run_split_hull_demo(mesh, animator=None):
    if (len(mesh.nodes) == 0):
        random_mesh_for_split_hull(mesh)
    if (len(mesh.edges) == 0):
        quick_hull(mesh)
    if (count_types(mesh) != 2):
        random_set_type(mesh, 2)
    mesh.remove_nodes_not_connected()
    run_demo_analyze('split_hull(mesh, 2, 4, animator)', mesh, animator)


def run_rand_multi_split_demo(mesh, animator=None):
    if (len(mesh.nodes) == 0):
        random_mesh_for_rand_multi_split(mesh)
    if (len(mesh.edges) == 0):
        quick_hull(mesh)
    if (count_types(mesh) < 3):
        random_set_type(mesh, 3)
    mesh.remove_nodes_not_connected()
    run_demo_analyze('rand_multi_split(mesh, animator)', mesh, animator)


def run_demo_analyze(str_function, mesh, animator=None):
    print('Running '+str_function)
    if (animator):
        animator.halt_skip()
        animator.set_mesh(mesh)
        animator.set_text(None)
        animator.wait()
    str_fcn_name = str_function[:str_function.index('(')]
    str_date_time = time.strftime("%Y%m%d%H%M%S")
    filename_mesh = str_fcn_name + '-' + str_date_time + '.mesh'
    filename_prof = str_fcn_name + '.prof'
    mesh.save(filename_mesh)
    cProfile.run(str_function, filename_prof)
    time.sleep(0.1)
    p = pstats.Stats(filename_prof)
    os.remove(filename_prof)
    p.strip_dirs().sort_stats('calls')
    regex_nobuiltin = '\A[^\{].*[^\}]\Z'
    regex_nothreading = '\A[^threading].*'
    p.print_stats(regex_nobuiltin, regex_nothreading)
    #p.print_callers()
    if (mesh.is_consistent()):
        os.remove(filename_mesh)
        print('Mesh is consistent.')
    else:
        print('ERROR: Mesh is inconsistent.')
        print('Input mesh saved at: ' + filename_mesh)


def get_refcounts():
    """
    Checks for memory leaks
    http://www.nightmare.com/medusa/memory-leaks.html
    may be obsolete, because Python > 2.0 checks for cyclic references
    """
    #gc.collect()
    d = {}
    sys.modules
    # collect all classes
    for m in sys.modules.values():
        for sym in dir(m):
            o = getattr (m, sym)
            if type(o) is types.ClassType:
                d[o] = sys.getrefcount (o)
    # sort by refcount
    pairs = map (lambda x: (x[1],x[0]), d.items())
    pairs.sort()
    pairs.reverse()
    return pairs


def print_refcounts(num_top):
    for n, c in get_refcounts()[:num_top]:
        print '%10d %s' % (n, c.__name__)


