from hcs.data.LinkedList import LinkedList
from hcs.data.AdjacencyListMesh import AdjacencyListMesh


__author__="Gernot WALZL"
__date__ ="2010-02-20"


class MyMesh(AdjacencyListMesh):
    """
    Dictionaries in python work like hash maps (O(1) query time)
    http://svn.python.org/view/python/tags/r264/Objects/dictnotes.txt

    The lists edges_open and edges_no_faces will only be updated when removing
    faces.
    """

    def __init__(self):
        AdjacencyListMesh.__init__(self)
        self.hash_nodes = dict()
        self.hash_edges = dict()
        self.hash_faces = dict()
        self.edges_open = LinkedList()
        self.edges_no_faces = LinkedList()


    def find_node(self, node):
        """
        O(1)
        """
        try:
            result = self.hash_nodes[node]
        except KeyError:
            result = None
        return result


    def find_edge(self, edge):
        """
        O(1)
        """
        try:
            result = self.hash_edges[edge]
        except KeyError:
            result = None
        return result


    def find_face(self, face):
        """
        O(1)
        """
        try:
            result = self.hash_faces[face]
        except KeyError:
            result = None
        return result


    def add_node(self, node):
        """
        O(1)
        """
        #if (self.find_node(node)):
        #    raise Exception('Node already exists.')
        self.lock.acquire()
        AdjacencyListMesh.add_node(self, node)
        self.hash_nodes[node] = node
        self.lock.release()


    def add_edge(self, edge):
        """
        O(1)
        """
        #if (self.find_edge(edge)):
        #    raise Exception('Edge already exists.')
        self.lock.acquire()
        AdjacencyListMesh.add_edge(self, edge)
        self.hash_edges[edge] = edge
        self.lock.release()


    def add_face(self, face, create_empty_edges=False):
        """
        O(1)
        """
        #if (self.find_face(face)):
        #    raise Exception('Face already exists.')
        self.lock.acquire()
        AdjacencyListMesh.add_face(self, face, create_empty_edges)
        self.hash_faces[face] = face
        self.lock.release()


    def remove_node(self, node):
        """
        O(1)
        """
        self.lock.acquire()
        AdjacencyListMesh.remove_node(self, node)
        del self.hash_nodes[node]
        self.lock.release()


    def remove_edge(self, edge):
        self.lock.acquire()
        AdjacencyListMesh.remove_edge(self, edge)
        if (len(self.edges_open) > 0):
            self.edges_open.remove(edge)
        if (len(self.edges_no_faces) > 0):
            self.edges_no_faces.remove(edge)
        del self.hash_edges[edge]
        self.lock.release()


    def remove_face(self, face):
        self.lock.acquire()
        AdjacencyListMesh.remove_face(self, face)
        for i in range(0, 3):
            if (i == 0):
                edge = face.edge_a
            elif (i == 1):
                edge = face.edge_b
            elif (i == 2):
                edge = face.edge_c
            if (edge):
                if (edge.face_r is None and edge.face_l is None):
                    self.edges_no_faces.add(edge)
                    self.edges_open.remove(edge)
                elif (edge.face_l is None):
                    self.edges_open.add(edge)
                elif (edge.face_r is None):
                    self.edges_open.add(edge)
        del self.hash_faces[face]
        self.lock.release()


    def handle_edges_open(self):
        """
        O(1)
        Removes all elements of edges_open.
        """
        self.lock.acquire()
        result = self.edges_open
        self.edges_open = LinkedList()
        self.lock.release()
        return result


    def remove_edges_no_faces(self):
        """
        O(len(self.edges_no_faces))
        """
        self.lock.acquire()
        for edge in self.edges_no_faces:
            self.remove_edge(edge)
        self.edges_no_faces.remove_all()
        self.lock.release()


    def is_consistent(self):
        result = True
        self.lock.acquire()
        if (not AdjacencyListMesh.is_consistent(self)):
            result = False
        if (len(self.nodes) == len(self.hash_nodes)):
            for node in self.nodes:
                if not node in self.hash_nodes:
                    result = False
                    break
        else:
            result = False
        if (len(self.edges) == len(self.hash_edges)):
            for edge in self.edges:
                if not edge in self.hash_edges:
                    result = False
                    break
        else:
            result = False
        if (len(self.faces) == len(self.hash_faces)):
            for face in self.faces:
                if not face in self.hash_faces:
                    result = False
                    break
        else:
            result = False
        self.lock.release()
        return result


    def destroy(self):
        self.lock.acquire()
        self.hash_nodes.clear()
        self.hash_edges.clear()
        self.hash_faces.clear()
        self.edges_open.destroy()
        self.edges_no_faces.destroy()
        AdjacencyListMesh.destroy(self)
        self.lock.release()


