from threading import RLock

__author__="Gernot WALZL"
__date__ ="2009-11-08"



class Element:

    def __init__(self, data):
        self.data = data
        self.prev = None
        self.next = None



class Iterator:

    def __init__(self, current_element):
        self.current_element = current_element

    def __iter__(self):
        return self

    def next(self):
        """
        O(1)
        """
        if (not self.current_element):
            raise StopIteration
        result = self.current_element.data
        self.current_element = self.current_element.next
        return result



class LinkedList:
    """
    This class implements an iterable, thread safe, double linked list.
    """

    def __init__(self):
        self.lock = RLock()
        self.lock.acquire()
        self.first_element = None
        self.last_element = None
        self.num_elements = 0
        self.lock.release()


    def __iter__(self):
        return Iterator(self.first_element)


    def __str__(self):
        self.lock.acquire()
        result = '[' + ', '.join([str(i) for i in iter(self)]) + ']'
        self.lock.release()
        return result


    def __len__(self):
        return self.num_elements


    def __getitem__(self, index):
        return self.get(index)


    def __add__(self, value):
        return self.add(value)


    def size(self):
        """
        O(n)
        """
        self.lock.acquire()
        i = 0
        current = self.first_element
        while (current):
            i += 1
            current = current.next
        self.lock.release()
        return i


    def add(self, value):
        """
        O(1)
        """
        self.lock.acquire()
        element = Element(value)
        if (not self.first_element):
            self.first_element = element
        if (self.last_element):
            self.last_element.next = element
            element.prev = self.last_element
        self.last_element = element
        self.num_elements += 1
        self.lock.release()
        return element


    def find_element(self, value):
        """
        O(n)
        """
        self.lock.acquire()
        result = None
        current = self.first_element
        while (current):
            if (current.data == value):
                result = current
                break
            current = current.next
        self.lock.release()
        return result


    def find_index(self, value):
        self.lock.acquire()
        result = -1
        i = 0
        current = self.first_element
        while (current):
            if (current.data == value):
                result = i
                break
            i += 1
            current = current.next
        self.lock.release()
        return result


    def remove_element(self, element):
        """
        O(1)
        """
        self.lock.acquire()
        if (element.prev):
            element.prev.next = element.next
        if (element.next):
            element.next.prev = element.prev
        if (element is self.first_element):
            self.first_element = element.next
        if (element is self.last_element):
            self.last_element = element.prev
        self.num_elements -= 1
        self.lock.release()


    def remove(self, value):
        """
        O(n)
        """
        self.lock.acquire()
        result = False
        element = self.find_element(value)
        if (element):
            self.remove_element(element)
            result = True
        self.lock.release()
        return result


    def get(self, index):
        """
        O(n)
        """
        self.lock.acquire()
        result = None
        current = self.first_element
        i = 0
        while (current):
            if (i == index):
                result = current.data
                break
            i += 1
            current = current.next
        self.lock.release()
        return result


    def set(self, index, value):
        """
        O(n)
        """
        self.lock.acquire()
        result = False
        i = 0
        current = self.first_element
        while (current):
            if (i == index):
                current.data = value
                result = True
                break
            i += 1
            current = current.next
        self.lock.release()
        return result


    def add_at(self, index, value):
        """
        O(n)
        """
        self.lock.acquire()
        if (index == 0):
            element = Element(value)
            if (self.first_element):
                self.first_element.prev = element
                element.next = self.first_element
            self.first_element = element
            self.num_elements += 1
        elif (self.num_elements <= index):
            element = self.add(value)
        else:
            i = 0
            current = self.first_element
            while (current):
                if (i == index):
                    element = Element(value)
                    element.next = current
                    element.prev = current.prev
                    current.prev.next = element
                    current.prev = element
                    self.num_elements += 1
                    break
                i += 1
                current = current.next
        self.lock.release()
        return element


    def remove_at(self, index):
        """
        O(n)
        """
        self.lock.acquire()
        result = False
        i = 0
        current = self.first_element
        while (current):
            if (i == index):
                self.remove_element(current)
                result = True
                break
            i += 1
            current = current.next
        self.lock.release()
        return result


    def contains(self, value):
        """
        O(n)
        """
        self.lock.acquire()
        result = (not self.find_element(value) is None)
        self.lock.release()
        return result


    def remove_all(self):
        """
        O(n)
        """
        self.lock.acquire()
        element = self.first_element
        while (element):
            self.remove_element(element)
            element = element.next
        self.first_element = None
        self.last_element = None
        self.num_elements = 0
        self.lock.release()


    def is_consistent(self):
        """
        O(n)
        """
        self.lock.acquire()
        result = True
        if (self.size() != self.num_elements):
            result = False
        element = self.first_element
        while (element):
            if (element.prev):
                if (not element.prev.next is element):
                    result = False
                    break
            else:
                if (not self.first_element is element):
                    result = False
                    break
            if (element.next):
                if (not element.next.prev is element):
                    result = False
                    break
            else:
                if (not self.last_element is element):
                    result = False
                    break
            element = element.next
        self.lock.release()
        return result


    def clone(self):
        """
        O(n)
        """
        result = self.__class__()
        self.lock.acquire()
        element = self.first_element
        while (element):
            result.add(element.data)
            element = element.next
        self.lock.release()
        return result


    def destroy(self):
        """
        O(n)
        Circular references may be bad for Pythons garbage collector.
        """
        self.lock.acquire()
        element = self.first_element
        while (element):
            current = element
            element = element.next
            current.data = None
            current.prev = None
            current.next = None
        self.first_element = None
        self.last_element = None
        self.num_elements = 0
        self.lock.release()


