from threading import RLock
from hcs.data.LinkedList import LinkedList
from hcs.data.Node import Node
from hcs.data.Node import NodeData
from hcs.data.Edge import Edge
from hcs.data.Edge import EdgeData
from hcs.data.Face import Face
from hcs.data.Face import FaceData


__author__="Gernot WALZL"
__date__ ="2009-11-08"


class AdjacencyListMesh:

    def __init__(self):
        self.lock = RLock()
        self.lock.acquire()
        self.description = ''
        self.nodes = LinkedList()
        self.edges = LinkedList()
        self.faces = LinkedList()
        self.lock.release()


    def __str__(self):
        result = 'nodes:\n'
        i = 0
        for node in self.nodes:
            result += '  [' + str(i) + ']=' + str(node) + '\n'
            if (len(node.edges) > 0):
                result += '    edges=['
                j = 0
                for edge in node.edges:
                    index = self.edges.find_index(edge)
                    if (j > 0):
                        result += ', '
                    result += 'edges[' + str(index) + ']'
                    j += 1
                result += ']\n'
            if (len(node.faces) > 0):
                result += '    faces=['
                j = 0
                for face in node.faces:
                    index = self.faces.find_index(face)
                    if (j > 0):
                        result += ', '
                    result += 'faces[' + str(index) + ']'
                    j += 1
                result += ']\n'
            i += 1
        result += 'edges:\n'
        i = 0
        for edge in self.edges:
            result += '  [' + str(i) + ']=' + str(edge) + '\n'
            i += 1
        result += 'faces:\n'
        i = 0
        for face in self.faces:
            result += '  [' + str(i) + ']=' + str(face) + '\n'
            i += 1
        return result


    def find_node(self, node):
        """
        O(len(self.nodes))
        """
        self.lock.acquire()
        result = None
        for current_node in self.nodes:
            if (current_node == node):
                result = current_node
                break
        self.lock.release()
        return result


    def find_edge(self, edge):
        """
        O(len(self.edges))
        """
        self.lock.acquire()
        result = None
        for current_edge in self.edges:
            if (current_edge == edge):
                result = current_edge
                break
        self.lock.release()
        return result


    def find_face(self, face):
        """
        O(len(self.faces))
        """
        self.lock.acquire()
        result = None
        for current_face in self.faces:
            if (current_face == face):
                result = current_face
                break
        self.lock.release()
        return result


    def add_node(self, node):
        """
        O(1)
        """
        self.lock.acquire()
        node.mesh = self
        node.mesh_list_element = self.nodes.add(node)
        self.lock.release()


    def add_edge(self, edge):
        """
        O(1)
        """
        self.lock.acquire()
        edge.mesh = self
        edge.mesh_list_element = self.edges.add(edge)
        edge.node_a_list_element = edge.node_a.edges.add(edge)
        edge.node_b_list_element = edge.node_b.edges.add(edge)
        self.lock.release()


    def add_face(self, face, create_empty_edges=False):
        """
        O(1)
        """
        self.lock.acquire()
        face.mesh = self
        face.mesh_list_element = self.faces.add(face)
        face.node_a_list_element = face.node_a.faces.add(face)
        face.node_b_list_element = face.node_b.faces.add(face)
        face.node_c_list_element = face.node_c.faces.add(face)
        if (create_empty_edges):
            if (not face.edge_a):
                face.edge_a = Edge(face.node_b, face.node_c)
                face.edge_a.face_l = face
                self.add_edge(face.edge_a)
            if (not face.edge_b):
                face.edge_b = Edge(face.node_c, face.node_a)
                face.edge_b.face_l = face
                self.add_edge(face.edge_b)
            if (not face.edge_c):
                face.edge_c = Edge(face.node_a, face.node_b)
                face.edge_c.face_l = face
                self.add_edge(face.edge_c)
        self.lock.release()


    def remove_node(self, node):
        """
        O(len(node.edges))
        """
        self.lock.acquire()
        for face in node.faces:
            self.remove_face(face)
        for edge in node.edges:
            self.remove_edge(edge)
        self.nodes.remove_element(node.mesh_list_element)
        node.mesh_list_element.data = None
        node.mesh_list_element = None
        node.mesh = None
        self.lock.release()


    def remove_edge(self, edge):
        """
        O(1)
        """
        self.lock.acquire()
        edge.node_a.edges.remove_element(edge.node_a_list_element)
        edge.node_a_list_element.data = None
        edge.node_a_list_element = None
        edge.node_b.edges.remove_element(edge.node_b_list_element)
        edge.node_b_list_element.data = None
        edge.node_b_list_element = None
        if (edge.face_l):
            self.remove_face(edge.face_l)
            edge.face_l = None
        if (edge.face_r):
            self.remove_face(edge.face_r)
            edge.face_r = None
        self.edges.remove_element(edge.mesh_list_element)
        edge.mesh_list_element.data = None
        edge.mesh_list_element = None
        edge.mesh = None
        self.lock.release()


    def remove_face(self, face):
        """
        O(1)
        """
        self.lock.acquire()
        for i in range(0, 3):
            if (i == 0):
                node = face.node_a
                node_list_element = face.node_a_list_element
            elif (i == 1):
                node = face.node_b
                node_list_element = face.node_b_list_element
            elif (i == 2):
                node = face.node_c
                node_list_element = face.node_c_list_element
            node.faces.remove_element(node_list_element)
            node_list_element.data = None
            node_list_element = None
        for i in range(0, 3):
            if (i == 0):
                edge = face.edge_a
            elif (i == 1):
                edge = face.edge_b
            elif (i == 2):
                edge = face.edge_c
            if (edge):
                if (edge.face_r is face):
                    edge.face_r = None
                elif (edge.face_l is face):
                    edge.face_l = edge.face_r
                    edge.face_r = None
                    edge.swap_nodes()
        self.faces.remove_element(face.mesh_list_element)
        face.mesh_list_element.data = None
        face.mesh_list_element = None
        face.mesh = None
        self.lock.release()


    def remove_edges_no_faces(self):
        """
        O(len(self.edges))
        """
        self.lock.acquire()
        for edge in self.edges:
            if (edge.face_l is None and edge.face_r is None):
                self.remove_edge(edge)
        self.lock.release()


    def remove_nodes_not_connected(self):
        """
        O(len(self.nodes))
        """
        self.lock.acquire()
        for node in self.nodes:
            if (len(node.edges) == 0 and len(node.faces) == 0):
                self.remove_node(node)
        self.lock.release()


    def union(self, other):
        """
        O(len(self.nodes) * len(other.nodes))
        """
        self.lock.acquire()
        for node in other.nodes:
            if (not self.find_node(node)):
                node.edges.remove_all()
                node.faces.remove_all()
                self.add_node(node)
        other.nodes.destroy()
        for edge in other.edges:
            if (not self.find_edge(edge)):
                edge.node_a = self.find_node(edge.node_a)
                edge.node_b = self.find_node(edge.node_b)
                self.add_edge(edge)
        other.edges.destroy()
        for face in other.faces:
            if (not self.find_face(face)):
                face.node_a = self.find_node(face.node_a)
                face.node_b = self.find_node(face.node_b)
                face.node_c = self.find_node(face.node_c)
                face.edge_a = self.find_edge(Edge(face.node_c, face.node_b))
                if (face.edge_a):
                    if (face.edge_a.node_a is face.node_b):
                        face.edge_a.face_l = face
                    else:
                        face.edge_a.face_r = face
                face.edge_b = self.find_edge(Edge(face.node_a, face.node_c))
                if (face.edge_b):
                    if (face.edge_b.node_a is face.node_c):
                        face.edge_b.face_l = face
                    else:
                        face.edge_b.face_r = face
                face.edge_c = self.find_edge(Edge(face.node_b, face.node_a))
                if (face.edge_c):
                    if (face.edge_c.node_a is face.node_a):
                        face.edge_c.face_l = face
                    else:
                        face.edge_c.face_r = face
                self.add_face(face, False)
        other.faces.destroy()
        self.lock.release()


    def save(self, filename):
        file = open(filename, 'w')
        self.lock.acquire()
        if (len(self.nodes) > 0):
            file.write('nodes:\n')
            i = 0
            for node in self.nodes:
                file.write('  [' + str(i) + ']=' + str(node) + '\n')
                i += 1
        if (len(self.edges) > 0):
            file.write('edges:\n')
            i = 0
            for edge in self.edges:
                file.write('  [' + str(i) + ']=' + str(edge) + '\n')
                i += 1
        if (len(self.faces) > 0):
            file.write('faces:\n')
            i = 0
            for face in self.faces:
                file.write('  [' + str(i) + ']=' + str(face) + '\n')
                i += 1
        self.lock.release()
        file.close()


    def load(self, filename):
        file = open(filename, 'r')
        self.lock.acquire()
        self.clear()
        state = 0
        error = False
        i = 0
        for line in file:
            i += 1
            if (line.startswith('nodes')):
                state = 1
                continue
            elif (line.startswith('edges')):
                state = 2
                continue
            elif (line.startswith('faces')):
                state = 3
                continue
            if (state == 1 or state == 2 or state == 3):
                index = int(line[line.index('[')+1:line.index(']')].strip())
                strobj = line[line.index('=')+1:].strip()
            if (state == 0):
                self.description += line
            elif (state == 1):
                if (index != len(self.nodes)):
                    error = True
                    break
                node = Node.from_str(strobj)
                if (self.find_node(node)):
                    error = True
                    break
                else:
                    self.add_node(node)
            elif (state == 2):
                if (index != len(self.edges)):
                    error = True
                    break
                edge = Edge.from_str(strobj, self)
                if (not self.find_edge(edge)):
                    if (edge.node_a and edge.node_b):
                        self.add_edge(edge)
                    else:
                        error = True
                        break
            elif (state == 3):
                if (index != len(self.faces)):
                    error = True
                    break
                face = Face.from_str(strobj, self)
                if (not self.find_face(face)):
                    if (face.node_a and face.node_b and face.node_c):
                        face.edge_a = self.find_edge(
                            Edge(face.node_c, face.node_b))
                        if (face.edge_a):
                            if (face.edge_a.node_a is face.node_b):
                                face.edge_a.face_l = face
                            else:
                                face.edge_a.face_r = face
                        face.edge_b = self.find_edge(
                            Edge(face.node_a, face.node_c))
                        if (face.edge_b):
                            if (face.edge_b.node_a is face.node_c):
                                face.edge_b.face_l = face
                            else:
                                face.edge_b.face_r = face
                        face.edge_c = self.find_edge(
                            Edge(face.node_b, face.node_a))
                        if (face.edge_c):
                            if (face.edge_c.node_a is face.node_a):
                                face.edge_c.face_l = face
                            else:
                                face.edge_c.face_r = face
                        self.add_face(face, False)
                    else:
                        error = True
                        break
        self.description = self.description[:len(self.description)-1]
        self.lock.release()
        file.close()
        if (error):
            raise Exception('Error in '+filename+':'+str(i)+': '+line)


    def clear(self):
        self.lock.acquire()
        self.description = ''
        for face in self.faces:
            self.remove_face(face)
        for edge in self.edges:
            self.remove_edge(edge)
        for node in self.nodes:
            self.remove_node(node)
        self.lock.release()


    def is_consistent(self):
        self.lock.acquire()
        result = True
        for node in self.nodes:
            if (node.mesh_list_element is None):
                result = False
                break
            for edge in node.edges:
                if (not self.edges.contains(edge)):
                    result = False
                    break
                if ((not edge.node_a is node) and
                        (not edge.node_b is node)):
                    result = False
                    break
            for face in node.faces:
                if (not self.faces.contains(face)):
                    result = False
                    break
                if ((not face.node_a is node) and
                        (not face.node_b is node) and
                        (not face.node_c is node)):
                    result = False
                    break
        for edge in self.edges:
            if (edge.mesh_list_element is None):
                result = False
                break
            if ((not self.nodes.contains(edge.node_a)) or
                    (not self.nodes.contains(edge.node_b))):
                result = False
                break
            if ((not edge.node_a.edges.contains(edge)) or
                    (not edge.node_b.edges.contains(edge))):
                result = False
                break
            if (edge.face_l):
                if (not self.faces.contains(face)):
                    result = False
                    break
                if ((not edge.face_l.edge_a is edge) and
                        (not edge.face_l.edge_b is edge) and
                        (not edge.face_l.edge_c is edge)):
                    result = False
                    break
            if (edge.face_r):
                if (not self.faces.contains(face)):
                    result = False
                    break
                if ((not edge.face_r.edge_a is edge) and
                        (not edge.face_r.edge_b is edge) and
                        (not edge.face_r.edge_c is edge)):
                    result = False
                    break
        for face in self.faces:
            if (face.mesh_list_element is None):
                result = False
                break
            if ((not self.nodes.contains(face.node_a)) or
                    (not self.nodes.contains(face.node_b)) or
                    (not self.nodes.contains(face.node_c))):
                result = False
                break
            if ((not face.node_a.faces.contains(face)) or
                    (not face.node_b.faces.contains(face)) or
                    (not face.node_c.faces.contains(face))):
                result = False
                break
            if ((not self.edges.contains(face.edge_a)) or
                    (not self.edges.contains(face.edge_b)) or
                    (not self.edges.contains(face.edge_c))):
                result = False
                break
            if ((not face.edge_a.face_l is face) and
                    (not face.edge_a.face_r is face)):
                result = False
                break
            if ((not face.edge_b.face_l is face) and
                    (not face.edge_b.face_r is face)):
                result = False
                break
            if ((not face.edge_c.face_l is face) and
                    (not face.edge_c.face_r is face)):
                result = False
                break
        self.lock.release()
        return result


    def clone(self):
        """
        O(len(self.nodes))
        """
        result = self.__class__()
        self.lock.acquire()
        for node in self.nodes:
            node_cloned = node.clone()
            node_cloned.data = NodeData()
            node_cloned.data.origin = node
            if (not node.data):
                node.data = NodeData()
            node.data.origin = node_cloned
            result.add_node(node_cloned)
        for edge in self.edges:
            edge_cloned = Edge(edge.node_a.data.origin, edge.node_b.data.origin)
            edge_cloned.data = EdgeData()
            edge_cloned.data.origin = edge
            if (not edge.data):
                edge.data = EdgeData()
            edge.data.origin = edge_cloned
            result.add_edge(edge_cloned)
        for face in self.faces:
            face_cloned = Face(face.node_a.data.origin,
                face.node_b.data.origin, face.node_c.data.origin)
            face_cloned.data = FaceData()
            face_cloned.data.origin = face
            if (not face.data):
                face.data = FaceData()
            face.data.origin = face_cloned
            if (face.edge_a):
                face_cloned.edge_a = face.edge_a.data.origin
            if (face.edge_b):
                face_cloned.edge_b = face.edge_b.data.origin
            if (face.edge_c):
                face_cloned.edge_c = face.edge_c.data.origin
            result.add_face(face_cloned)
        for edge in self.edges:
            if (edge.face_l):
                edge.data.origin.face_l = edge.face_l.data.origin
            if (edge.face_r):
                edge.data.origin.face_r = edge.face_r.data.origin
        for node in self.nodes:
            node.data.origin = None
        for edge in self.edges:
            edge.data.origin = None
        for face in self.faces:
            face.data.origin = None
        self.lock.release()
        return result


    def destroy(self):
        """
        O(len(self.nodes))
        Circular references may be bad for Pythons garbage collector.
        """
        self.lock.acquire()
        self.description = ''
        for face in self.faces:
            face.node_a = None
            face.node_b = None
            face.node_c = None
            face.edge_a = None
            face.edge_b = None
            face.edge_c = None
            face.list_element = None
            face.node_a_list_element = None
            face.node_b_list_element = None
            face.node_c_list_element = None
            face.highlight = False
            face.data = None
        self.faces.destroy()
        for edge in self.edges:
            edge.node_a = None
            edge.node_b = None
            edge.face_l = None
            edge.face_r = None
            edge.list_element = None
            edge.node_a_list_element = None
            edge.node_b_list_element = None
            edge.highlight = False
            edge.data = None
        self.edges.destroy()
        for node in self.nodes:
            node.position = None
            node.edges.destroy()
            node.faces.destroy()
            node.list_element = None
            node.type = 0
            node.highlight = False
            node.data = None
        self.nodes.destroy()
        self.lock.release()


