import math
from hcs.data.Edge import Edge
from hcs.data.Face import Face
from hcs.data.LinkedList import LinkedList


__author__="Gernot WALZL"
__date__ ="2010-03-02"


def distance_node(node_a, node_b):
    distance = 0.0
    for i in range(0,3):
        distance += (node_a.position[i] - node_b.position[i])**2
    distance = math.sqrt(distance)
    return distance


def cross_prod(a=[0.0, 0.0, 0.0], b=[0.0, 0.0, 0.0]):
    result = [0.0, 0.0, 0.0]
    result[0] = a[1]*b[2] - a[2]*b[1]
    result[1] = a[2]*b[0] - a[0]*b[2]
    result[2] = a[0]*b[1] - a[1]*b[0]
    return result


def distance_edge(edge, node):
    """
    Geradengleichung: x = p_gerade + t * a
    x = <x, y, z>; a ... Richtung der Geraden
    d(p) = | (a / |a|) x (p - p_gerade) |  (p_gerade ... Punkt auf Gerade)
    """
    a = [0.0, 0.0, 0.0]
    length_a = 0.0
    for i in range(0, 3):
        a[i] = edge.node_a.position[i] - edge.node_b.position[i]
        length_a += a[i]**2
    length_a = math.sqrt(length_a)
    for i in range(0, 3):
        a[i] /= length_a

    b = [0.0, 0.0, 0.0]
    for i in range(0, 3):
        b[i] = node.position[i] - edge.node_a.position[i]

    c = cross_prod(a, b)
    distance = 0.0
    for i in range(0, 3):
        distance += c[i]**2
    distance = math.sqrt(distance)

    return distance


def distance_face(face, node):
    """
    Ebenengleichung: u*x + v*y + w*z = c
    n = <u, v, w>; x = <x, y, z>
    n*x = c = n*p_ebene   (p_ebene ... Punkt auf Ebene)
    d(p) = |n*p - c| / |n|
    """
    c = 0.0
    n = face.normal()
    for i in range(0, 3):
        c += face.node_a.position[i] * n[i]
    result = -c
    dn = 0.0
    for i in range(0, 3):
        result += n[i] * node.position[i]
        dn += n[i]**2
    dn = math.sqrt(dn)
    result = result / dn
    return result


def is_face_visible(face, node):
    """
    O(1)
    """
    result = False
    if (not (node is face.node_a or
            node is face.node_b or
            node is face.node_c)):
        if (distance_face(face, node) > 0.0):
            result = True
    return result


def get_visible_faces_recursive(result, face_hint, node_visible):
    if (face_hint.highlight):
        return
    if (not is_face_visible(face_hint, node_visible)):
        return
    if (face_hint.edge_a.face_l is face_hint):
        face_a = face_hint.edge_a.face_r
    else:
        face_a = face_hint.edge_a.face_l
    if (face_hint.edge_b.face_l is face_hint):
        face_b = face_hint.edge_b.face_r
    else:
        face_b = face_hint.edge_b.face_l
    if (face_hint.edge_c.face_l is face_hint):
        face_c = face_hint.edge_c.face_r
    else:
        face_c = face_hint.edge_c.face_l
    result.add(face_hint)
    face_hint.highlight = True
    if (face_a):
        get_visible_faces_recursive(result, face_a, node_visible)
    if (face_b):
        get_visible_faces_recursive(result, face_b, node_visible)
    if (face_c):
        get_visible_faces_recursive(result, face_c, node_visible)


def triangulate_edges_with_node(mesh, edges, node):
    """
    O(len(node.edges)**2)
    """
    for edge in edges:
        face = Face(edge.node_b, edge.node_a, node)
        face.edge_c = edge
        edge.face_r = face
        edge_create = Edge(edge.node_a, node)
        face.edge_a = edge_create
        edge_create.face_l = face
        mesh.add_edge(edge_create)
        mesh.add_face(face, False)
    for face in node.faces:
        for edge in node.edges:
            if (face.node_a is edge.node_a):
                face.edge_b = edge
                edge.face_r = face
                break


def get_neighbour_points(node):
    """
    O(len(node.edges))
    """
    result = LinkedList()
    for edge in node.edges:
        if (edge.node_a is node):
            result.add(edge.node_b)
        elif (edge.node_b is node):
            result.add(edge.node_a)
    return result


def count_types(mesh):
    """
    O(len(mesh.nodes))
    """
    num_types = 0
    for node in mesh.nodes:
        if (node.type > num_types):
            num_types = node.type
    return num_types


