from hcs.data.LinkedList import LinkedList
from hcs.data.Node import NodeData
from hcs.data.Edge import Edge
from hcs.data.Face import Face
from hcs.data.MyMesh import MyMesh
from hcs.algo.common import is_face_visible
from hcs.algo.common import get_neighbour_points
from hcs.algo.common import get_visible_faces_recursive
from hcs.algo.common import triangulate_edges_with_node
from hcs.algo.QuickHull import quick_hull


__author__="Gernot WALZL"
__date__ ="2009-11-08"


def split_hull(mesh, blue=2, d0=6, animator=None):
    split_hull = SplitHull(mesh, blue, d0, animator)
    split_hull.run()



class SHNodeData(NodeData):

    def __init__(self):
        NodeData.__init__(self)
        self.red_points_list_element = None



class SplitHull:
    """
    1. If P contains no red points, return conv(P).
    2. If there exists a red point r in P for which we have
       deg_P r <= d_0 (with a suitable constant d_0), then return
       SplitHull(conv(P \ r)).
    3. Take random blue points b until (i) deg_P(b) <= 6;
       and (ii) there exists a blue edge e in conv(P \ b) visible
       from b.
    4. Call SplitHull(conv(P \ b)) to compute conv(B \ b).
    5. Using e as a starting edge, insert b into conv(B \ b)
       and return conv(B).
    """

    def __init__(self, my_mesh, blue=2, d0=6, animator=None):
        self.mesh = my_mesh
        self.blue = blue
        self.d0 = d0
        self.animator = animator
        self.red_points_small_deg = LinkedList()
        self.num_red_points = 0


    def init_lists(self):
        """
        O(len(self.mesh.nodes))
        """
        for node in self.mesh.nodes:
            if (not node.data):
                node.data = SHNodeData()
            if (node.type != self.blue):
                self.num_red_points += 1
                if (len(node.edges) <= self.d0):
                    node.data.red_points_list_element = \
                        self.red_points_small_deg.add(node)


    def update_list_red_points(self, nodes):
        """
        O(len(nodes))
        """
        for node in nodes:
            if (node.data.red_points_list_element):
                if (len(node.edges) > self.d0):
                    self.red_points_small_deg.remove_element(
                        node.data.red_points_list_element)
                    node.data.red_points_list_element = None
            elif (node.type != self.blue):
                if (len(node.edges) <= self.d0):
                    node.data.red_points_list_element = \
                        self.red_points_small_deg.add(node)


    def remove_red_point(self, node_r):
        """
        O(len(node_r.edges))
        """
        self.mesh.remove_node(node_r)
        self.red_points_small_deg.remove_element(
            node_r.data.red_points_list_element)
        node_r.data.red_points_list_element = None
        self.num_red_points -= 1
        return


    def get_points_from_edges(self, edges):
        """
        O(len(edges))
        """
        result = LinkedList()
        for edge in edges:
            result.add(edge.node_a)
        return result


    def conv_part(self, nodes, node_visible):
        """
        O(len(nodes) * log(len(nodes)))
        """
        my_mesh = MyMesh()
        for node in nodes:
            my_mesh.add_node(node.clone())
        quick_hull(my_mesh)
        for face in my_mesh.faces:
            if (not is_face_visible(face, node_visible)):
                my_mesh.remove_face(face)
        my_mesh.remove_edges_no_faces()
        return my_mesh


    def triangulate_open_edges(self, node_visible):
        """
        O(len(nodes) * len(self.mesh.nodes))
        """
        edges = self.mesh.handle_edges_open()
        nodes = self.get_points_from_edges(edges)
        if (len(edges) == 3):
            face = Face(nodes[0], nodes[1], nodes[2])
            face.edge_a = edges[1]
            face.edge_a.face_r = face
            face.edge_b = edges[2]
            face.edge_b.face_r = face
            face.edge_c = edges[0]
            face.edge_c.face_r = face
            if (not is_face_visible(face, node_visible)):
                face.swap_normal()
            self.mesh.add_face(face)
        elif (len(edges) > 3):
            my_mesh = self.conv_part(nodes, node_visible)
            self.mesh.union(my_mesh)
            my_mesh.destroy()
        edges.destroy()
        nodes.destroy()


    def get_visible_edge(self, face, node):
        """
        O(1)
        """
        result = None
        if (face.node_a is node or
                face.node_b is node or
                face.node_c is node):
            if ((not face.edge_a.node_a is node) and
                    (not face.edge_a.node_b is node)):
                result = face.edge_a
            if ((not face.edge_b.node_a is node) and
                    (not face.edge_b.node_b is node)):
                result = face.edge_b
            if ((not face.edge_c.node_a is node) and
                    (not face.edge_c.node_b is node)):
                result = face.edge_c
        return result


    def find_visible_blue_edge_1(self, node):
        """
        O(len(node.edges))
        """
        result = None
        for face in node.faces:
            edge = self.get_visible_edge(face, node)
            if (edge):
                if (edge.get_type() == self.blue):
                    result = edge
                    break
        return result


    def find_correct_blue_point_1(self):
        result = None
        for node in self.mesh.nodes:
            if (node.type == self.blue):
                if (len(node.edges) <= 6):
                    if (self.find_visible_blue_edge_1(node)):
                        result = node
                        break
        return result


    def find_visible_blue_edge_2(self, node):
        result = None
        self.mesh.remove_node(node)
        edges = self.mesh.handle_edges_open()
        nodes = self.get_points_from_edges(edges)
        my_mesh = self.conv_part(nodes, node)
        for edge in my_mesh.edges:
            if (edge.get_type() == self.blue):
                result = Edge(edge.node_a.clone(), edge.node_b.clone())
                break
        my_mesh.destroy()
        self.mesh.add_node(node)
        triangulate_edges_with_node(self.mesh, edges, node)
        nodes.destroy()
        edges.destroy()
        return result


    def find_correct_blue_point_2(self):
        result = None
        num_nodes = len(self.mesh.nodes)
        i_node = 0
        for node in self.mesh.nodes:
            if (node.type == self.blue):
                num_edges = len(node.edges)
                if (4 <= num_edges and num_edges <= 6):
                    if (self.find_visible_blue_edge_2(node)):
                        result = node
                        break
            i_node += 1
            if (num_nodes <= i_node):
                break
        return result


    def remove_all_visible_faces(self, face_hint, node_visible):
        visible_faces = LinkedList()
        get_visible_faces_recursive(visible_faces, face_hint, node_visible)
        for face in visible_faces:
            self.mesh.remove_face(face)
        visible_faces.destroy()


    def insert_node(self, my_edge, node):
        if (is_face_visible(my_edge.face_l, node)):
            face_hint = my_edge.face_l
        else:
            face_hint = my_edge.face_r
        self.remove_all_visible_faces(face_hint, node)
        self.mesh.remove_edges_no_faces()
        if (self.animator):
            self.animator.wait()
        edges = self.mesh.handle_edges_open()
        triangulate_edges_with_node(self.mesh, edges, node)
        edges.destroy()


    def split(self):
        """
        O(len(self.mesh.nodes))
        """

        # 1. If P contains no red points, return conv(P).
        if (self.num_red_points == 0):
            return

        # 2. If there exists a red point r in P for which we have
        # deg_P r <= d_0 (with a suitable constant d_0), then return
        # SplitHull(conv(P\r)).
        if (self.red_points_small_deg.first_element):
            node_r = self.red_points_small_deg.first_element.data
            if (self.animator):
                node_r.highlight = True
                self.animator.wait()
            nodes_neighbour = get_neighbour_points(node_r)
            self.remove_red_point(node_r)
            if (self.animator):
                self.animator.wait()
            if (len(self.mesh.nodes) > 2):
                self.triangulate_open_edges(node_r)
            self.update_list_red_points(nodes_neighbour)
            nodes_neighbour.destroy()
            if (self.animator):
                self.animator.wait()
            self.split()

        # FIX for Algorithm
        if (self.num_red_points == 0):
            return

        # 3. Take random blue points b until (i) deg_P(b) <= 6;
        # and (ii) there exists a blue edge e in conv(P\b) visible
        # from b.
        node_b = self.find_correct_blue_point_1()
        if (node_b):
            edge_b = self.find_visible_blue_edge_1(node_b)
            added_edge = False
        else:
            node_b = self.find_correct_blue_point_2()
            if (node_b):
                edge_b = self.find_visible_blue_edge_2(node_b)
                self.mesh.add_edge(edge_b)
                added_edge = True
            else:
                raise Exception('No useable blue point found with given d0. '+
                    'You need to increase d0.')

        if (self.animator):
            node_b.highlight = True
            edge_b.highlight = True
            self.animator.wait()

        # 4. Call SplitHull(conv(P\b)) to compute conv(B\b).
        nodes_neighbour = get_neighbour_points(node_b)
        self.mesh.remove_node(node_b)

        if (self.animator):
            node_b.highlight = False
            self.animator.wait()

        if (added_edge):
            self.mesh.remove_edge(edge_b)
        self.triangulate_open_edges(node_b)
        self.update_list_red_points(nodes_neighbour)
        nodes_neighbour.destroy()

        if (self.animator):
            edge_b.highlight = False
            self.animator.wait()

        self.split()

        # 5. Using e as a starting edge, insert b into conv(B\b)
        # and return conv(B).
        self.mesh.add_node(node_b)
        edge_b = self.mesh.find_edge(edge_b)

        if (self.animator):
            node_b.highlight = True
            edge_b.highlight = True
            self.animator.wait()

        self.insert_node(edge_b, node_b)

        if (self.animator):
            self.animator.wait()
            node_b.highlight = False
            edge_b.highlight = False


    def run(self):
        """
        O(len(self.mesh.nodes))
        """
        if (self.animator):
            self.animator.set_text('SplitHull')
            self.animator.wait()
        self.init_lists()
        self.split()


