from hcs.data.LinkedList import LinkedList
from hcs.data.Edge import Edge
from hcs.data.Face import Face
from hcs.data.Face import FaceData
from hcs.algo.common import distance_node
from hcs.algo.common import distance_edge
from hcs.algo.common import distance_face
from hcs.algo.common import is_face_visible
from hcs.algo.common import get_visible_faces_recursive
from hcs.algo.common import triangulate_edges_with_node


__author__="Gernot WALZL"
__date__ ="2009-11-08"


def quick_hull(mesh, animator=None):
    quick_hull = QuickHull(mesh, animator)
    quick_hull.run()



class QHFaceData(FaceData):

    def __init__(self, possible_points):
        FaceData.__init__(self)
        self.possible_points = possible_points
        self.incomplete_list_element = None



class QuickHull:

    def __init__(self, my_mesh, animator=None):
        self.mesh = my_mesh
        self.animator = animator
        self.incomplete_faces = LinkedList()


    def get_point_lowest(self):
        """
        O(len(self.mesh.nodes))
        """
        result = None
        for node in self.mesh.nodes:
            if (not result):
                result = node
                continue
            if (result.position[2] > node.position[2]):
                result = node
        return result


    def get_point_max_dist_node(self, my_node):
        """
        O(len(self.mesh.nodes))
        """
        result = None
        max_dist = 0.0
        for node in self.mesh.nodes:
            if (node is my_node):
                continue
            dist = distance_node(my_node, node)
            if (dist > max_dist):
                result = node
                max_dist = dist
        return result


    def get_point_max_dist_edge(self, edge):
        """
        O(len(self.mesh.nodes))
        """
        result = None
        max_dist = 0.0
        for node in self.mesh.nodes:
            if (node is edge.node_a or node is edge.node_b):
                continue
            dist = distance_edge(edge, node)
            if (dist > max_dist):
                result = node
                max_dist = dist
        return result


    def get_point_max_dist_face(self, face):
        """
        O(len(self.mesh.nodes))
        """
        result = None
        max_dist = 0.0
        for node in self.mesh.nodes:
            if (node is face.node_a or
                    node is face.node_b or
                    node is face.node_c):
                continue
            dist = distance_face(face, node)
            if (dist > max_dist):
                result = node
                max_dist = dist
        return result


    def init_polygon(self):
        """
        O(len(self.mesh.nodes))
        """
        if (len(self.mesh.nodes) < 4):
            raise Exception('There are at least 4 points required '+
                'to compute the convex hull.')

        node_a = self.get_point_lowest()
        if (self.animator):
            node_a.highlight = True
            self.animator.wait()
        node_b = self.get_point_max_dist_node(node_a)
        if (self.animator):
            node_b.highlight = True
            self.animator.wait()
        edge = Edge(node_a, node_b)
        node_c = self.get_point_max_dist_edge(edge)
        if (self.animator):
            node_c.highlight = True
            self.animator.wait()

        face_a = Face(node_a, node_b, node_c)
        node_da = self.get_point_max_dist_face(face_a)
        distance_a = 0.0
        if (node_da):
            distance_a = distance_face(face_a, node_da)
        face_b = Face(node_a, node_c, node_b)
        node_db = self.get_point_max_dist_face(face_b)
        distance_b = 0.0
        if (node_db):
            distance_b = distance_face(face_b, node_db)
        if (distance_a > distance_b):
            face = face_b    # correct orientation = outside positive
            node_d = node_da
        else:
            face = face_a    # correct orientation = outside positive
            node_d = node_db

        if (self.animator):
            node_d.highlight = True
            self.animator.wait()
            node_a.highlight = False
            node_b.highlight = False
            node_c.highlight = False
            node_d.highlight = False

        self.mesh.add_face(face, True)
        face_a = Face(face.node_c, face.node_b, node_d)
        face_a.edge_c = face.edge_a
        face_a.edge_c.face_r = face_a
        self.mesh.add_face(face_a, True)
        face_b = Face(face.node_a, face.node_c, node_d)
        face_b.edge_c = face.edge_b
        face_b.edge_c.face_r = face_b
        face_b.edge_a = face_a.edge_b
        face_b.edge_a.face_r = face_b
        self.mesh.add_face(face_b, True)
        face_c = Face(face.node_b, face.node_a, node_d)
        face_c.edge_c = face.edge_c
        face_c.edge_c.face_r = face_c
        face_c.edge_a = face_b.edge_b
        face_c.edge_a.face_r = face_c
        face_c.edge_b = face_a.edge_a
        face_c.edge_b.face_r = face_c
        self.mesh.add_face(face_c, True)


    def update_incomplete_faces(self, faces, possible_points):
        """
        O(len(faces) * len(possible_points))
        """
        for face in faces:
            face_possible_points = LinkedList()
            for node in possible_points:
                if (is_face_visible(face, node)):
                    face_possible_points.add(node)
            if (len(face_possible_points) > 0):
                face.data = QHFaceData(face_possible_points)
                face.data.incomplete_list_element = self.incomplete_faces.add(
                    face)


    def init_incomplete_faces(self):
        self.update_incomplete_faces(self.mesh.faces, self.mesh.nodes)


    def get_possible_point_max_dist_face(self, face, possible_points):
        """
        O(len(possible_points))
        """
        result = None
        max_dist = 0.0
        for node in possible_points:
            dist = distance_face(face, node)
            if (dist > max_dist):
                result = node
                max_dist = dist
        return result


    def remove_visible_faces(self, face_hint, node):
        """
        O(len(visible_faces))
        """
        possible_points = LinkedList()
        visible_faces = LinkedList()
        face_hint.highlight = False
        get_visible_faces_recursive(visible_faces, face_hint, node)
        if (self.animator):
            self.animator.wait()
        for face in visible_faces:
            if (face.data):
                for node in face.data.possible_points:
                    possible_points.add(node)
                face.data.possible_points.destroy()
                self.incomplete_faces.remove_element(
                    face.data.incomplete_list_element)
                face.data.incomplete_list_element = None
                face.data = None
            self.mesh.remove_face(face)
        self.mesh.remove_edges_no_faces()
        visible_faces.destroy()
        return possible_points


    def handle_face(self, face):
        """
        O(len(possible_points))
        """
        possible_points = face.data.possible_points
        node = self.get_possible_point_max_dist_face(face, possible_points)
        if (self.animator):
            node.highlight = True
            self.animator.wait()
        possible_points = self.remove_visible_faces(face, node)
        if (self.animator):
            self.animator.wait()
        edges_open = self.mesh.handle_edges_open()
        triangulate_edges_with_node(self.mesh, edges_open, node)
        edges_open.destroy()
        self.update_incomplete_faces(node.faces, possible_points)
        possible_points.destroy()
        if (self.animator):
            self.animator.wait()
            node.highlight = False


    def run(self):
        """
        O(len(self.mesh.nodes) * log(len(self.mesh.nodes)))
        """
        if (self.animator):
            self.animator.set_text('QuickHull')
            self.animator.wait()
        self.init_polygon()
        self.init_incomplete_faces()
        if (self.animator):
            self.animator.wait()
        element = self.incomplete_faces.first_element
        while (element):   # use linked list as queue
            face = element.data
            if (self.animator):
                face.highlight = True
                self.animator.wait()
            self.handle_face(face)
            self.incomplete_faces.remove_element(element)
            element = self.incomplete_faces.first_element


